<?php

namespace App\Integrations;

use App\Models\SalesChannelType;
use Exception;
use Generator;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Support\Facades\File;

/**
 * Class Channel.
 */
abstract class Channel
{
    /**
     * Namespace of mock sales channel classes.
     */
    const MOCK_NAMESPACE = 'App\\MockSalesChannels';

    /**
     * @return Channel
     *
     * @throws Exception
     */
    public static function getInstance(string $type, array $credentials)
    {
        if (! in_array($type, SalesChannelType::TYPES)) {
            throw new Exception('Channel type not found');
        }

        $namespace = app()->environment() === 'testing' ? self::MOCK_NAMESPACE : __NAMESPACE__;
        $channel_class = $namespace.'\\'.$type;

        return new $channel_class($credentials);
    }

    /**
     * Functions for all sales channels.
     */

    /**
     * Get Orders from sales channel.
     *
     * @param  null  $options
     * @return Generator
     */
    abstract public function getSalesOrders($options = null);

    /**
     * Get Products from sales channel.
     *
     * @param  null  $options
     * @return Generator
     */
    abstract public function getProducts($options = null);

    /**
     * Check sales channel credentials.
     *
     * @return mixed
     */
    abstract public function checkCredentials();

    /**
     * Read json file of Mocks resources.
     *
     *
     * @return array
     *
     * @throws FileNotFoundException
     *
     * @example $this->mocks('amazon.orders');
     */
    protected function mocks(string $path)
    {
        $path = resource_path('mocks/'.str_replace('.', '/', $path).'.json');

        return json_decode(File::get($path), true);
    }
}
