<?php

declare(strict_types=1);

namespace App\Integrations;

use App\Data\ApiLogData;
use App\Models\ApiLog;
use App\Models\IntegrationInstance;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Str;

/**
 * Class ApiLogger
 */
class ApiLogger
{
    /**
     * @param IntegrationInstance $integrationInstance
     * @param string $url
     * @param string $requestBody
     * @param Response|null $response
     * @param array $requestHeaders
     * @param int|null $statusCode
     * @return ApiLog|Model
     */
    public function logApiCall(
        IntegrationInstance $integrationInstance,
        string $url,
        string $requestBody,
        ?Response $response = null,
        array $requestHeaders = [],
        ?int $statusCode = null
    ): ApiLog|Model {
        /** @var ApiLog $apiLog */
        return ApiLog::query()->create([
            'integration_instance_id' => $integrationInstance->id,
            'url' => $url,
            'requestHeaders' => $requestHeaders,
            'requestBody' => $requestBody,
            'responseStatusCode' => $statusCode ?? $response?->status(),
            'responseHeaders' => $response?->headers(),
            'responseBody' => $response?->body(),
        ]);
    }

    /**
     * This method logs an API call asynchronously.
     *
     * @param IntegrationInstance $integrationInstance
     * @param ApiLogData $data
     * @return string
     */
    public function logApiCallAsync(
        IntegrationInstance $integrationInstance,
        ApiLogData $data
    ): string {

        // Generate a UUID for the log
        $uuid = (string) Str::uuid();

        // Dispatch the job to log the API call
        dispatch(function() use ($integrationInstance, $data, $uuid){
            ApiLog::query()->create([
                'integration_instance_id' => $integrationInstance->id,
                'url' => $data->url,
                'requestHeaders' => $data->requestHeaders,
                'requestBody' => $data->requestBody,
                'responseStatusCode' => $data->statusCode ?? @$data->response['status'],
                'responseHeaders' => @$data->response['headers'],
                'responseBody' => @$data->response['body'],
                'uuid' => $uuid,
            ]);
        });

        return $uuid;
    }
}
