<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 12/14/20
 * Time: 1:54 PM.
 */

namespace App\Importers;

use App\Exceptions\ImportFailedException;
use App\Importers\Parsers\FieldParserFactory;
use App\Lib\DataList\DataListFactory;
use App\Models\TaskStatus\TaskStatus;
use Exception;

/**
 * Class FileImporter.
 */
abstract class FileImporter
{
    const FILE_FORMAT_CSV = 'csv';

    const FILE_FORMAT_XLSX = 'xlsx';

    const FILE_FORMAT_TXT = 'txt';

    const SUPPORTED_FORMATS = [
        self::FILE_FORMAT_CSV,
        self::FILE_FORMAT_XLSX,
        self::FILE_FORMAT_TXT,
    ];

    protected $inBackground = false;

    /**
     * @var TaskStatus
     */
    protected $taskStatus;

    /**
     * @var string
     */
    protected $filePath;

    /**
     * Array of records to be imported.
     *
     * @var array
     */
    protected $records;

    /**
     * Records count by incrementing through processing rows.
     *
     * @var int
     */
    public $recordsCountByLooping = 0;

    /**
     * Array of headers in the file.
     *
     * @var array
     */
    protected $headers;

    /**
     * Array of required columns in the import.
     *
     * @var array
     */
    protected $requiredColumns = [];

    /**
     * Array of expected columns.
     *
     * @var array
     */
    protected $expectedColumns = [];

    /**
     * @var string
     */
    protected $idColumn = 'id';

    /**
     * @var array A map indicating a re-write of the fields in the file.
     */
    protected $mapping = [];

    /**
     * @var array An array of fields and their re-write rules
     */
    protected $reWrites = [];

    /**
     * @var array An array of validation errors
     */
    protected $validationErrors = [];

    /** @var \App\Lib\DataList\DataListInterface */
    protected $dataList;

    /**
     * FileImporter constructor.
     */
    public function __construct($task, string $filePath)
    {
        $this->dataList = DataListFactory::load($filePath);

        $this->initializeImportResources($filePath, $task);
    }

    /**
     * @throws ImportFailedException
     */
    public function import(): bool
    {
        try {
            $this->getStatusReporter()->addMessage('Import started at '.date("H:i:s \o\\n F jS, Y"));
            $this->getStatusReporter()->status = TaskStatus::STATUS_STARTED;

            //  Open file and look for headers
            if (empty($this->headers)) {
                $this->extractHeaders();
            }

            // Register the number of columns to be imported
            $this->getStatusReporter()->addMessage(count($this->headers).' columns.');

            //  If a required column is missing, complain, set status to 'Failed' and abort
            //  (in the future we could show a column mapping dialog at this point)
            if (! $this->validateRequired()) {
                throw new ImportFailedException('Required columns are missing: '.
                    implode(', ', $this->absentRequired()));
            }

            //  Report the total number of entries and show the progress bar at 0
            $this->handleBeginImportStatuses();

            // We load the records
            //      if (empty($this->records)) {
            //        $this->loadRecords();
            //      }
            //  Perform the updates and advance the progress bar
            return $this->importByRow();
        } catch (Exception $e) {
            report($e);
            $this->getStatusReporter()->addErrorMessage($e->getMessage());
            $this->getStatusReporter()->addMessage('Import aborted. '.$e->getFile().' '.$e->getLine());
            $this->getStatusReporter()->status = TaskStatus::STATUS_FAILED;
            $this->getStatusReporter()->save();

            // User::find($this->user_id)->notify(new BackgroundTaskCompleted($this->task_id, $this->user_id, "Listing importing job failed."));
            return false; // Failed
        }
    }

    public function setMapping(array $mapping)
    {
        $this->mapping = $mapping;
    }

    /**
     * @throws ImportFailedException
     */
    public function getHeaders(): array
    {
        if (empty($this->headers)) {
            $this->extractHeaders();
        }

        return $this->headers;
    }

    protected function findMappedValue($fileField)
    {
        $match = array_values(array_filter($this->mapping, function ($map) use ($fileField) {
            return $map['file_field'] === $fileField;
        }));

        return count($match) > 0 ? $match[0]['expected_field'] : null;
    }

    /**
     * @return \Generator
     *
     * @throws ImportFailedException
     */
    protected function yieldRecords()
    {
        if (empty($this->headers)) {
            $this->extractHeaders();
        }

        foreach ($this->dataList->getRows() as $parsed) {
            if ($parsed) {
                $combined = array_combine($this->headers, $parsed);
                $filtered = [];
                foreach ($combined as $k => $v) {
                    if (in_array($k, $this->expectedColumns)) {
                        if (is_string($v)) {
                            $v = trim(str_replace('"', '', preg_replace('/\s+/', ' ', $v)));
                            if (trim($v) == '') {
                                $v = null; // Force empty string to null.
                            }

                            if ($this->hasReWrites($k)) {
                                $filtered[$k] = $this->reWrite($k, $v);
                            } else {
                                $filtered[$k] = $v;
                            }
                        } else {
                            $filtered[$k] = $v;
                        }
                    }
                }

                yield $filtered;
            }
        }
    }

    /**
     * @throws Exception
     */
    protected function importByRow(): bool
    {
        try {
            $this->beforeImport();

            $errorCount = 0;

            // We read the file in chunks and import
            // the records right away. This prevents
            // memory issues for large files as we
            // won't be storing the entire content
            // of the file in an array.
            $records = $this->records ?? $this->yieldRecords();

            foreach ($records as $record) {
                try {
                    // Attempt to import the row
                    $this->importRow($record);
                    $this->getStatusReporter()->progressAdvance();
                } catch (Exception $e) {
                    $errorCount++;
                    $this->getStatusReporter()->addErrorMessage($e->getMessage());
                    $this->getStatusReporter()->addErrorMessage($e->getFile().' '.$e->getLine());
                }

                $this->recordsCountByLooping += 1;
            }

            if ($errorCount) {
                $this->getStatusReporter()->addErrorMessage($errorCount.' records with errors.');

                return false;
            }
            $this->finalizeImport();
            $this->getStatusReporter()->addMessage('Import completed.');

            //  When finished, remove the progress bar and set status to "Completed"
            $this->afterImportStatus();
        } catch (Exception $e) {
            $this->importErrorStatus($e->getMessage());

            if (config('app.env') === 'testing') {
                throw $e;
            }
            return false; // Failed
        }

        return true; // Success
    }

    public function runInBackground()
    {
        $this->inBackground = true;

        return $this;
    }

    protected function afterImportStatus()
    {
        $this->getStatusReporter()->progress = null;
        $this->getStatusReporter()->status = TaskStatus::STATUS_COMPLETED;
    }

    protected function importErrorStatus($error)
    {
        $this->getStatusReporter()->addErrorMessage($error);
        $this->getStatusReporter()->addMessage('Import aborted.');
        $this->getStatusReporter()->status = TaskStatus::STATUS_FAILED;
    }

    /**
     * Handles statuses before import begins.
     */
    protected function handleBeginImportStatuses()
    {
        $totalRecords = $this->getRecordsCount();
        $this->getStatusReporter()->addMessage($totalRecords.' entries.');

        //  Count duplicate records and report
        //    $uniqueIds = $this->countUniqueIds();
        //    if ($uniqueIds < $totalRecords) {
        //      $this->getStatusReporter()->addErrorMessage(($totalRecords - $uniqueIds).' duplicate IDs. ('.$uniqueIds.' unique records)');
        //    }

        $this->getStatusReporter()->addMessage('Processing entries...');
        $this->getStatusReporter()->progress_start = 0;
        $this->getStatusReporter()->progress_end = $totalRecords;
        $this->getStatusReporter()->progress = 0;
        $this->getStatusReporter()->save();
    }

    /**
     * Counts the unique ids in the import.
     *
     * @return int
     */
    protected function countUniqueIds()
    {
        // Returns the count of records removing duplicates (by id)
        if (empty($this->headers)) {
            $this->extractHeaders();
        }

        $ids = array_map(function ($x) {
            return $x[$this->idColumn];
        }, $this->records);

        $unique_ids = array_unique($ids);

        return count($unique_ids);
    }

    /**
     * Extracts headers from the import file.
     *
     * @return $this|FileImporter
     *
     * @throws ImportFailedException
     */
    protected function extractHeaders()
    {
        try {
            $headers = $this->dataList->getColumnHeadings();
            $headers = array_map('trim', $headers);
        } catch (Exception $e) {
            throw new ImportFailedException(
                'Failed to obtain column headers from uploaded file.',
                previous: $e
            );
        }

        if (! empty($headers)) {
            if (! empty($this->mapping)) {
                // Re-write applicable headers with values in mapping
                $tempHeaders = [];
                foreach ($headers as $header) {
                    $mapping = $this->findMappedValue($header);
                    $tempHeaders[] = $mapping ?? $header;
                }

                $this->headers = $tempHeaders;
            } else {
                $this->headers = $headers;
            }
        } else {
            if (empty($this->headers)) {
                throw new ImportFailedException('Headers could not be extracted.');
            }
        }

        return $this;
    }

    /**
     * Initializes the file import resources.
     */
    protected function initializeImportResources(string $filePath, $task)
    {
        if ($task) {
            $this->setTask($task);
        }

        // Prepare the file
        $this->prepareFile($filePath);
    }

    public function setTask($task)
    {
        if (is_null($task)) {
            $this->taskStatus = new TaskStatus();
        } else {
            $this->taskStatus = $task instanceof TaskStatus ? $task : TaskStatus::with([])->findOrNew($task);
        }
        $this->taskStatus->save();
    }

    /**
     * @throws ImportFailedException
     */
    protected function prepareFile($path): void
    {
        if (! is_readable($path)) {
            throw new ImportFailedException('Uploaded file could not be read.');
        }

        $this->filePath = $path;
    }

    /**
     * Loads the records to be imported into memory.
     *
     * @return $this
     */
    protected function loadRecords()
    {
        if (empty($this->headers)) {
            $this->extractHeaders();
        }
        $this->records = [];

        foreach ($this->dataList->getRows() as $parsed) {
            if ($parsed && count($this->headers) == count($parsed)) {
                $combined = array_combine($this->headers, $parsed);
                $filtered = [];
                foreach ($combined as $k => $v) {
                    if (in_array($k, $this->expectedColumns)) {
                        if (is_string($v)) {
                            $v = trim(str_replace('"', '', preg_replace('/\s+/', ' ', $v)));
                            if (trim($v) == '') {
                                $v = null; // Force empty string to null.
                            }

                            if ($this->hasReWrites($k)) {
                                $filtered[$k] = $this->reWrite($k, $v);
                            } else {
                                $filtered[$k] = $v;
                            }
                        } else {
                            $filtered[$k] = $v;
                        }
                    }
                }

                $this->records[] = $filtered;
            }
        }

        return $this;
    }

    /**
     * @return null
     *
     * @throws Exception
     */
    protected function reWrite($field, $value = null)
    {
        if (is_null($value)) {
            return $value;
        }

        // We get the re-write rules for the field
        $rules = $this->makeRewriteRulesForField($field);
        $parsed = $value;
        foreach ($rules as $rule) {
            $parsed = $rule->parse($parsed);
        }

        return $parsed;
    }

    /**
     * @return array
     *
     * @throws Exception
     */
    protected function makeRewriteRulesForField($field)
    {
        $fieldRewrites = collect($this->reWrites)->filter(function ($row) use ($field) {
            return $row['field'] === $field;
        })->toArray();

        if (count($fieldRewrites) === 0) {
            return [];
        }

        $results = [];

        foreach ($fieldRewrites as $parsers) {
            foreach ($parsers['rules'] as $parser) {
                $results[] = FieldParserFactory::make($parser['rule'], $parser['args']);
            }
        }

        return $results;
    }

    /**
     * Indicates if the given field has re-write rules.
     */
    protected function hasReWrites($field): bool
    {
        if (empty($this->reWrites)) {
            return false;
        }

        return in_array($field, collect($this->reWrites)->pluck('field')->toArray());
    }

    public function setRewrites(array $reWrites): static
    {
        $this->reWrites = $reWrites;

        return $this;
    }

    public function getRecordsCount(): int
    {
        return $this->dataList->getRowCount();
    }

    /**
     * @return array
     */
    public function getRecords()
    {
        if (empty($this->records)) {
            $this->loadRecords();
        }

        return $this->records;
    }

    public function getRequiredColumns(): array
    {
        return $this->requiredColumns;
    }

    /**
     * Validates required fields for the import.
     *
     * @return bool
     */
    protected function validateRequired()
    {
        return empty($this->absentRequired());
    }

    /**
     * Gets required fields that are absent.
     *
     * @return array
     */
    protected function absentRequired()
    {
        // Returns a list of missing required fields according to already
        // extracted headers
        if (empty($this->headers)) {
            $this->extractHeaders();
        }

        return array_diff($this->requiredColumns, $this->headers);
    }

    /**
     * @return array
     */
    public function getValidationErrors()
    {
        return $this->validationErrors;
    }

    protected function getStatusReporter()
    {
        if (empty($this->taskStatus)) {
            $this->setTask(null);
        }

        return $this->taskStatus;
    }

    /**
     * Imports a given row of data at a time.
     *
     *
     * @return mixed
     */
    abstract protected function importRow(array $row);

    abstract protected function beforeImport(): void;
    abstract protected function finalizeImport();
}
