<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Importers\DataImporter;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Repositories\SupplierInventoryRepository;
use App\Repositories\WarehouseRepository;
use App\Response;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

class WarehouseDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'name',
        'type',
    ];

    /**
     * @var WarehouseRepository
     */
    protected $warehouses;

    /**
     * @var SupplierInventoryRepository
     */
    protected $inventories;

    /**
     * WarehouseDataImporter constructor.
     *
     *
     * @throws BindingResolutionException
     */
    public function __construct($taskId, string $filePath)
    {
        parent::__construct($taskId, $filePath);

        $this->warehouses = app()->make(WarehouseRepository::class);
        $this->inventories = app()->make(SupplierInventoryRepository::class);
    }

    /**
     * @return mixed|void
     */
    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            // We create the given warehouse or update it if possible
            // We attempt to find by id or name
            if (! empty($row['id'])) {
                $warehouse = Warehouse::with([])->find($row['id']);
                if (! $warehouse) {
                    $this->validationErrors[$row['name'] ?? $row['id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $row['id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'name']));
                    $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                    return;
                }
            } else {
                $warehouse = Warehouse::with([])->where('name', $row['name'])
                    ->where('type', $row['type'])
                    ->first();
            }

            if (! $warehouse) {
                // If supplier name is available, we find the supplier
                // and add in the id to make the new warehouse (to be created) a supplier warehouse
                if (! empty($row['supplier'])) {
                    $supplier = Supplier::with([])->where('name', e($row['supplier']))->first();
                    if ($supplier) {
                        $row['supplier_id'] = $supplier->id;
                    }
                }
                $warehouse = $this->warehouses->createWarehouse($row);
                if ($warehouse->isSupplierWarehouse() && isset($row['is_default']) && $row['is_default']) {
                    // This must be the default warehouse of the supplier
                    $warehouse->supplier->setDefaultWarehouse($warehouse->id);

                    // If the supplier warehouse doesn't have inventory initiated, we do that now.
                    if ($warehouse->supplierInventory()->count() === 0) {
                        $this->inventories->initializeInventoryForSupplierWarehouse($warehouse);
                    }
                }
            }
            $warehouse->fill($row);
            $warehouse->save();

            // Set the warehouse address
            if (! $warehouse->wasRecentlyCreated) {
                $this->warehouses->setWarehouseAddress($warehouse, $row);
            }
        });
    }

    /**
     * Makes exportable fields.
     */
    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'name' => Exporter::makeExportableField('name', true, 'Warehouse Name'),
            'address_name' => Exporter::makeExportableField('address_name', true, 'Warehouse Address Name'),
            'type' => Exporter::makeExportableField('type'),
            'company' => Exporter::makeExportableField('company', true, 'Company Name'),
            'email' => Exporter::makeExportableField('email', true, 'Contact Email'),
            'phone' => Exporter::makeExportableField('phone', true, 'Phone Number'),
            'fax' => Exporter::makeExportableField('fax'),
            'address1' => Exporter::makeExportableField('address1', true, 'Address Line 1'),
            'address2' => Exporter::makeExportableField('address2', true, 'Address Line 2'),
            'address3' => Exporter::makeExportableField('address3', true, 'Address Line 3'),
            'city' => Exporter::makeExportableField('city'),
            'province' => Exporter::makeExportableField('province'),
            'province_code' => Exporter::makeExportableField('province_code'),
            'zip' => Exporter::makeExportableField('zip'),
            'country' => Exporter::makeExportableField('country'),
            'country_code' => Exporter::makeExportableField('country_code'),
            'supplier' => Exporter::makeExportableField('supplier_name', true, 'Supplier Name'),
            'order_fulfillment' => Exporter::makeExportableField('order_fulfillment'),
            'dropship_enabled' => Exporter::makeExportableField('dropship_enabled', true, 'Has Dropship Enabled (0 or 1)'),
            'direct_returns' => Exporter::makeExportableField('direct_returns', true, 'Has Direct Returns (0 or 1)'),
            'customer_returns' => Exporter::makeExportableField('customer_returns', true, 'Has Customer Returns (0 or 1)'),
            'is_default' => Exporter::makeExportableField('is_default', true, 'Is Default Warehouse (0 or 1)'),
        ];
    }
}
