<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter;
use App\Importers\DataImporter;
use App\Models\ProductCategory;
use App\Response;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

/**
 * Class ProductCategoryDataImporter.
 */
class ProductCategoryDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'name',
    ];

    /**
     * @return mixed|void
     */
    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            // We create the given category or update it if possible
            // We attempt to find by id or name
            if (! empty($row['id'])) {
                $category = ProductCategory::with([])->find($row['id']);
                if (! $category) {
                    $this->validationErrors[$row['name'] ?? $row['id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $row['id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'name']));
                    $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                    return;
                }
            } else {
                $category = ProductCategory::with([])->where('name', $row['name'])
                    ->first();
            }
            if (! $category) {
                $category = new ProductCategory;
                $category->is_leaf = true;
            }

            if (! $row['parent_id'] || ($category->wasRecentlyCreated)) {
                $row['parent_id'] = null;
            }

            $category->fill($row);
            $category->save();
        });
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => DataTableExporter::makeExportableField('id', true, 'ID'),
            'name' => DataTableExporter::makeExportableField('name'),
            'parent_id' => DataTableExporter::makeExportableField('parent_id', true, 'Parent Category ID'),
        ];
    }
}
