<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Importers\DataImporter;
use App\Models\Address;
use App\Models\Customer;
use App\Models\SalesChannel;
use App\Response;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

class CustomerDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'name',
    ];

    /**
     * @return mixed|void
     */
    protected function importRow(array $row)
    {
        if (! empty($row['id'])) {
            $customer = Customer::with([])->find($row['id']);
            if (! $customer) {
                $this->validationErrors[$row['name'] ?? $row['id']] = Response::getError("ID {$row['id']} doesn't exist in SKU, review file and import again.", Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'name']));
                $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                return;
            }
        } else {
            // Try to find by the
            $customer = Customer::exists($row);
        }

        if (! $customer) {
            $customer = new Customer();
            $address = new Address($row);
            $address->save();
            $customer->default_billing_address_id = $address->id;
            $customer->default_shipping_address_id = $address->id;
        }

        if (! empty($row['sales_channel_origin_name'])) {
            $salesChannel = SalesChannel::with([])
                ->whereHas('integrationInstance', function (Builder $builder) use ($row) {
                    $builder->where('name', $row['sales_channel_origin_name']);
                })
                ->first();
            if ($salesChannel) {
                $row['sales_channel_origin_id'] = $salesChannel->id;
            }
        }

        $customer->fill($row);
        $customer->save();

        // Set Address
        if (! ($address = $customer->address)) {
            $address = new Address($row);
        }
        $address->label = $row['label'] ?? 'Default Address';
        $address->fill($row);
        $address->save();
        if (! $customer->default_shipping_address_id) {
            $customer->default_shipping_address_id = $address->id;
        }

        // Set billing address
        $billingAddressFields = [];
        foreach ($row as $key => $value) {
            if (Str::startsWith($key, 'default_billing_address.')) {
                $parts = explode('.', $key);
                $billingAddressFields[$parts[1]] = $value;
            }
        }
        if (! empty($billingAddressFields)) {
            if (! ($billingAddress = $customer->billingAddress)) {
                $billingAddress = new Address($billingAddressFields);
                $billingAddress->label = $row['default_billing_address.label'] ?? 'Billing Address';
            } else {
                $billingAddress->fill($billingAddressFields);
            }
            $billingAddress->save();
            $customer->default_billing_address_id = $billingAddress->id;
        }

        // Save the customer
        $customer->save();
    }

    /**
     * Fields for export.
     */
    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'name' => Exporter::makeExportableField('name', true),
            'email' => Exporter::makeExportableField('email', true),
            'company_name' => Exporter::makeExportableField('company', true),
            'label' => Exporter::makeExportableField('label', true, 'Address Label'),
            'phone' => Exporter::makeExportableField('phone', true, 'Phone Number'),
            'fax' => Exporter::makeExportableField('fax', true),
            'address1' => Exporter::makeExportableField('address1', true, 'Address Line 1'),
            'address2' => Exporter::makeExportableField('address2', true, 'Address Line 2'),
            'address3' => Exporter::makeExportableField('address3', true, 'Address Line 3'),
            'city' => Exporter::makeExportableField('city', true),
            'province' => Exporter::makeExportableField('province', true),
            'province_code' => Exporter::makeExportableField('province_code', true),
            'zip' => Exporter::makeExportableField('zip', true),
            'country' => Exporter::makeExportableField('country', true),
            'country_code' => Exporter::makeExportableField('country_code', true),
            'default_billing_address.name' => Exporter::makeExportableField('default_billing_address.name', true),
            'default_billing_address.label' => Exporter::makeExportableField('default_billing_address.label', true),
            'default_billing_address.company' => Exporter::makeExportableField('default_billing_address.company', true),
            'default_billing_address.email' => Exporter::makeExportableField('default_billing_address.email', true),
            'default_billing_address.phone' => Exporter::makeExportableField('default_billing_address.phone', true),
            'default_billing_address.fax' => Exporter::makeExportableField('default_billing_address.fax', true),
            'default_billing_address.address1' => Exporter::makeExportableField('default_billing_address.address1', true),
            'default_billing_address.address2' => Exporter::makeExportableField('default_billing_address.address2', true),
            'default_billing_address.address3' => Exporter::makeExportableField('default_billing_address.address3', true),
            'default_billing_address.city' => Exporter::makeExportableField('default_billing_address.city', true),
            'default_billing_address.province' => Exporter::makeExportableField('default_billing_address.province', true),
            'default_billing_address.province_code' => Exporter::makeExportableField('default_billing_address.province_code', true),
            'default_billing_address.zip' => Exporter::makeExportableField('default_billing_address.zip', true),
            'default_billing_address.country' => Exporter::makeExportableField('default_billing_address.company', true),
            'default_billing_address.country_code' => Exporter::makeExportableField('default_billing_address.country_code', true),
            'default_warehouse.name' => Exporter::makeExportableField('default_warehouse_name', false),
            'default_warehouse.id' => Exporter::makeExportableField('warehouse_id', true, 'Default Warehouse ID'),
            'sales_channel_origin.name' => Exporter::makeExportableField('sales_channel_origin_name', false),
            'default_pricing_tier.name' => Exporter::makeExportableField('default_pricing_tier_name', false),
            'default_store.name' => Exporter::makeExportableField('default_store_name', false),
            'default_shipping_method_domestic.name' => Exporter::makeExportableField('default_shipping_method_domestic', false),
            'default_shipping_method_international.name' => Exporter::makeExportableField('default_shipping_method_international', false),
        ];
    }
}
