<?php

namespace App\Http\Resources;

use App\DataTable\DataTableResource;
use App\Models\WarehouseTransfer;
use Illuminate\Http\Request;

/**
 * Class WarehouseTransferResource.
 *
 * @mixin WarehouseTransfer
 */
class WarehouseTransferResource extends DataTableResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'warehouse_transfer_number' => $this->warehouse_transfer_number,
            'transfer_date' => $this->inclusive('transfer_date', $this->transfer_date),
            //      'item_quantity'          => $this->inclusive('item_quantity', $this->quantity),
            //      'item_sku'               => $this->whenLoaded('product', function(){ return $this->product->sku; }),
            //      'item_name'               => $this->whenLoaded('product', function(){ return $this->product->name; }),
            'from_warehouse' => $this->whenLoaded('fromWarehouse', function () {
                return [
                    'id' => $this->fromWarehouse->id,
                    'name' => $this->fromWarehouse->name,
                    'type' => $this->fromWarehouse->type,
                    'integration_instance_id' => $this->fromWarehouse->integration_instance_id,
                ];
            }),
            'to_warehouse' => $this->whenLoaded('toWarehouse', function () {
                return [
                    'id' => $this->toWarehouse->id,
                    'name' => $this->toWarehouse->name,
                    'type' => $this->toWarehouse->type,
                    $this->mergeWhen($this->whenLoaded('toWarehouse.address'), function () {
                        return ['address' => $this->toWarehouse->address?->only(['id', 'email'])];
                    })
                ];
            }),
            //      'receipt_date'      => $this->whenLoaded( 'warehouseTransferReceipt', function () { return $this->warehouseTransferReceipt->receipt_date; } ),
            'status' => $this->inclusive('status', $this->transfer_status),
            'shipment_status' => $this->inclusive('shipment_status', $this->shipment_status),
            'shipment_percent' => $this->shipment_percent,
            'shipment_id' => $this->shipment_id,
            'receipt_status' => $this->inclusive('receipt_status', $this->receipt_status),
            'receipt_percent' => $this->receipt_percent,
            'shipping_method' => $this->whenLoaded('shippingMethod', function () {
                return ShippingMethodResource::make($this->shippingMethod);
            }),
            'fully_shipped_at' => $this->inclusive('fully_shipped_at', $this->fully_shipped_at),
            'fully_received_at' => $this->inclusive('fully_received_at', $this->fully_received_at),
            'items' => $this->inclusive('items', $this->whenLoaded('warehouseTransferLines', function () {
                return WarehouseTransferLineResource::collection($this->warehouseTransferLines);
            })),
            'warehouse_transfer_shipment' => $this->whenLoaded('shipment', function () {
                return WarehouseTransferShipmentResource::make($this->shipment);
            }),
            'warehouse_transfer_shipment_receipts' => $this->whenLoaded('shipmentReceipts', function () {
                return WarehouseTransferReceiptResource::collection($this->shipmentReceipts);
            }),
            'eta' => $this->eta,
            'created_at' => $this->inclusive('created_at', $this->created_at),
            'updated_at' => $this->inclusive('updated_at', $this->updated_at),
            'archived_at' => $this->inclusive('archived_at', $this->archived_at),
            'document' => $this->inclusive('document', $this->document()),
            'notes' => $this->inclusive('notes', $this->whenLoaded('notes', function () {
                return $this->notes()
                    ->select('notes.id', 'notes.note', 'notes.created_at')
                    ->selectRaw('IFNULL(users.name, "System") as user_name')
                    ->leftJoin('users', 'users.id', '=', 'notes.user_id')
                    ->orderByDesc('created_at')
                    ->get();
            })),

            //-----------------------------------------------------
            // Linked Adjustments
            //-----------------------------------------------------
            'adjustments' => $this->inclusive('adjustments', $this->whenLoaded('warehouseTransferLines', function(){
                return InventoryAdjustmentResource::collection($this->warehouseTransferLines->pluck('adjustments')->flatten());
            })),
            'asn_last_sent_at' => $this->inclusive('asn_last_sent_at', $this->asn_last_sent_at),
            $this->mergeWhen($this->relationLoaded('shipments'), function () {
                return ['shipments' => $this->shipments];
            }),
        ];
    }
}
