<?php

namespace App\Http\Resources;

use App\DataTable\DataTableResource;
use App\Models\AccountingTransaction;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use Illuminate\Http\Request;

/**
 * Class SalesOrderResource.
 *
 *
 * @mixin SalesOrder
 */
class SalesOrderResource extends DataTableResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        return [
            //-----------------------------------------------------
            // Basic Info
            //-----------------------------------------------------
            'id' => $this->id,
            'sales_order_number' => $this->inclusive('sales_order_number', $this->sales_order_number),
            'sales_channel_id' => $this->sales_channel_id,
            'sales_channel_name' => $this->inclusive('sales_channel_name', $this->whenLoaded('salesChannel', function () {
                return $this->salesChannel->name;
            })),
            'integration_instance_id' => $this->inclusive('integration_instance_id', $this->whenLoaded('salesChannel', function () {
                return $this->salesChannel->integration_instance_id;
            })),
            'integration' => $this->inclusive('integration', $this->whenLoaded('salesChannel', function () {
                return $this->salesChannel->integration->name;
            })),
            'integration_id' => $this->inclusive('integration', $this->whenLoaded('salesChannel', function () {
                return $this->salesChannel->integration->id;
            })),
            'integration_image' => $this->inclusive('integration_image', $this->whenLoaded('salesChannel', function () {
                return $this->salesChannel->integration->image_url;
            })),
            'store' => $this->inclusive('store', $this->whenLoaded('store', function () {
                return $this->active_store?->only('id', 'name');
            })),
            'accounting' => [
                'transaction' => $this->whenLoaded('accountingTransaction', function () {
                    return AccountingTransactionResource::make($this->accountingTransaction);
                }),
            ],
            'order_date' => $this->inclusive('order_date', $this->order_date),
            'payment_date' => $this->inclusive('payment_date', $this->fully_paid_at),
            'created_at' => $this->inclusive('created_at', $this->created_at),
            'updated_at' => $this->inclusive('updated_at', $this->updated_at),
            'archived_at' => $this->inclusive('archived_at', $this->archived_at),
            'canceled_at' => $this->inclusive('canceled_at', $this->canceled_at),
            'currency' => $this->inclusive('currency', $this->currency->code),
            'currency_rate' => $this->inclusive('currency_rate', $this->currency_rate),
            'discount_lines' => $this->inclusive('discount_lines', $this->discount_lines),
            'tax_lines' => $this->inclusive('tax_lines', $this->tax_lines),
            'packing_slip_printed_at' => $this->inclusive('packing_slip_printed_at', $this->packing_slip_printed_at),
            'is_fba' => $this->is_fba,
            'is_replacement_order' => $this->is_replacement_order,
            'need_mapping' => $this->need_mapping,

            //-----------------------------------------------------
            // Fulfillment
            //-----------------------------------------------------

            'ship_by_date' => $this->inclusive('ship_by_date', $this->ship_by_date),
            'deliver_by_date' => $this->inclusive('deliver_by_date', $this->deliver_by_date),
            'fulfilled_at' => $this->inclusive('fulfilled_at', $this->fulfilled_at),
            'shipping_method_id' => $this->inclusive('shipping_method_id', $this->shipping_method_id),
            'shipping_method' => $this->inclusive('shipping_method', $this->shipping_method_name),
            'requested_shipping_method' => $this->inclusive('requested_shipping_method', $this->requested_shipping_method),
            'mapped_shipping_method' => $this->inclusive('mapped_shipping_method', $this->mapped_shipping_method),
            'has_backorder' => $this->inclusive('has_backorder', $this->whenLoaded('salesOrderLines.backorderQueue', function () {
                return $this->whereHas('salesOrderLines.backorderQueue', function ($query) {
                    $query->where('shortage_quantity', '>', 0);
                })->exists();
            })),
            'fulfillment' => $this->inclusive('fulfillment', SalesOrderFulfillmentResource::collection($this->whenLoaded('salesOrderFulfillments'))),

            'fulfillment_channel' => $this->inclusive('fulfillment_channel', $this->fulfillment_channel),
            //-----------------------------------------------------
            // Customer
            //-----------------------------------------------------

            $this->mergeWhen($this->whenLoaded('customer', true, false) && $this->customer->relationLoaded('address'), function () {
                return SalesCustomerAddressResource::make($this->customer);
            }),

            $this->mergeWhen($this->whenLoaded('customer', true, false) && $this->customer->relationLoaded('billingAddress'), function () {
                return [
                    'customer_billing_address' => [
                        'id' => $this->customer->default_billing_address_id ?? null,
                        'name' => $this->customer->billingAddress->name ?? null,
                        'email' => $this->customer->billingAddress->email ?? null,
                        'company' => $this->customer->billingAddress->company ?? null,
                        'phone' => $this->customer->billingAddress->phone ?? null,
                        'address1' => $this->customer->billingAddress->address1 ?? null,
                        'address2' => $this->customer->billingAddress->address2 ?? null,
                        'address3' => $this->customer->billingAddress->address3 ?? null,
                        'city' => $this->customer->billingAddress->city ?? null,
                        'province' => $this->customer->billingAddress->province ?? null,
                        'province_code' => $this->customer->billingAddress->province_code ?? null,
                        'zip' => $this->customer->billingAddress->zip ?? null,
                        'country' => $this->customer->billingAddress->country ?? null,
                        'country_code' => $this->customer->billingAddress->country_code ?? null,
                    ],
                ];
            }),

            //-----------------------------------------------------
            // Shipping Address
            //-----------------------------------------------------

            $this->mergeWhen($this->resource->relationLoaded('shippingAddress'), function () {
                return [
                    'sales_order_shipping_address' => [
                        'id' => $this->shipping_address_id,
                        'name' => $this->shippingAddress->name ?? null,
                        'email' => $this->shippingAddress->email ?? null,
                        'company' => $this->shippingAddress->company ?? null,
                        'phone' => $this->shippingAddress->phone ?? null,
                        'address1' => $this->shippingAddress->address1 ?? null,
                        'address2' => $this->shippingAddress->address2 ?? null,
                        'address3' => $this->shippingAddress->address3 ?? null,
                        'city' => $this->shippingAddress->city ?? null,
                        'province' => $this->shippingAddress->province ?? null,
                        'province_code' => $this->shippingAddress->province_code ?? null,
                        'zip' => $this->shippingAddress->zip ?? null,
                        'country' => $this->shippingAddress->country ?? null,
                        'country_code' => $this->shippingAddress->country_code ?? null,
                    ],
                ];
            }),

            //-----------------------------------------------------
            // Billing Address
            //-----------------------------------------------------

            $this->mergeWhen($this->resource->relationLoaded('billingAddress'), function () {
                return [
                    'sales_order_billing_address' => [
                        'id' => $this->billing_address_id,
                        'name' => $this->billingAddress->name ?? null,
                        'email' => $this->billingAddress->email ?? null,
                        'company' => $this->billingAddress->company ?? null,
                        'phone' => $this->billingAddress->phone ?? null,
                        'address1' => $this->billingAddress->address1 ?? null,
                        'address2' => $this->billingAddress->address2 ?? null,
                        'address3' => $this->billingAddress->address3 ?? null,
                        'city' => $this->billingAddress->city ?? null,
                        'province' => $this->billingAddress->province ?? null,
                        'province_code' => $this->billingAddress->province_code ?? null,
                        'zip' => $this->billingAddress->zip ?? null,
                        'country' => $this->billingAddress->country ?? null,
                        'country_code' => $this->billingAddress->country_code ?? null,
                    ],
                ];
            }),

            //-----------------------------------------------------
            // Items
            //-----------------------------------------------------

            'item_info' => $this->inclusive('item_info', $this->whenLoaded('salesOrderLines', function () {
                $items = $this->salesOrderLines->map->setRelation('salesOrder', $this->resource);

                return SalesOrderLineResource::collection($items);
            })),

            'purchase_orders' => $this->whenLoaded('purchaseOrders', function () {
                return $this->purchaseOrders->map->only('id', 'purchase_order_number', 'supplier_warehouse_id');
            }),

            //-----------------------------------------------------
            // Linked Adjustments
            //-----------------------------------------------------
            'adjustments' => $this->inclusive('adjustments', $this->whenLoaded(
                'adjustments',
                fn () => InventoryAdjustmentResource::collection($this->adjustments->toBase()->merge($this->salesOrderLines->pluck('adjustments')->flatten())))
            ),

            /*
            |--------------------------------------------------------------------------
            | Financial Lines
            |--------------------------------------------------------------------------
            */

            'financial_lines' => $this->whenLoaded('financialLines', function () {
                return $this->financialLines->map(function ($financialLine) {
                    return array_merge(
                        $financialLine->only(
                            'id',
                            'description',
                            'quantity',
                            'amount',
                            'amount_in_tenant_currency',
                            'tax_allocation',
                            'tax_allocation_in_tenant_currency',
                            'tax_rate_id',
                            'allocate_to_products',
                            'proration_strategy',
                            'allocate_to_id',
                            'allocate_to_type'
                        ),
                        [
                            'financial_line_type' => $financialLine->financialLineType->only('id', 'name', 'classification', 'nominal_code_id'),
                            'nominal_code' => $financialLine->nominalCode?->only('id', 'name', 'code'),
                        ]
                    );
                });
            }),

            /*
            |--------------------------------------------------------------------------
            | Custom Fields
            |--------------------------------------------------------------------------
            */
            'custom_field_values' => $this->whenLoaded('customFieldValues', function () {
                return $this->customFieldValues->map(function ($customFieldValue) {
                    return array_merge($customFieldValue->only(
                        'id',
                        'value'
                    ), [
                        'name' => $customFieldValue->customField->name,
                    ]);
                });
            }),

            //-----------------------------------------------------
            // Statuses
            //-----------------------------------------------------

            'order_status' => $this->inclusive('order_status', $this->order_status),
            'fulfillment_status' => $this->inclusive('fulfillment_status', $this->fulfillment_status),
            'fulfillment_sequence' => $this->inclusive('fulfillment_sequence', $this->salesOrderFulfillments()->latest('fulfillment_sequence')->value('fulfillment_sequence')),
            'payment_status' => $this->inclusive('payment_status', $this->payment_status),

            //-----------------------------------------------------
            // Totals
            //-----------------------------------------------------

            $this->mergeWhen($this->relationLoaded('salesOrderLineFinancials'), function () {
                return [
                    'total_revenue' => $this->inclusive('total_revenue', $this->getFinancialTotal('total_revenue')),
                    'total_revenue_allocated' => $this->inclusive('total_revenue_allocated', $this->getFinancialTotal('revenue_allocated')),
                    'financial_lines_revenue_total' => $this->inclusive('financial_lines_revenue_total', $this->getFinancialLinesRevenueTotal()),
                    // To be used in the future if we want this accessible, add data table config
                    'financial_lines_revenue_allocatable' => $this->inclusive('financial_lines_revenue_allocatable', $this->getFinancialLinesRevenueTotal(true)),
                    'financial_lines_revenue_not_allocatable' => $this->inclusive('financial_lines_revenue_not_allocatable', $this->getFinancialLinesRevenueTotal(false)),
                ];
            }),

            $this->mergeWhen($this->relationLoaded('salesOrderLines'), function () {
                return [
                    'tax_revenue' => $this->inclusive('tax_revenue', $this->getTaxRevenue()),
                    'product_total' => $this->inclusive('product_total', $this->product_subtotal),
                    'total' => $this->inclusive('total', $this->total),
                    'discount_total' => $this->inclusive('discount_total', $this->calculated_discount_total),
                ];
            }),
            'currency_code' => $this->inclusive('currency_code', $this->whenLoaded('currency', function () {
                return $this->currency->code ?? null;
            })),

            //-----------------------------------------------------
            // Other
            //-----------------------------------------------------

            'tags' => $this->inclusive('tags', $this->whenLoaded('tags', fn () => $this->tags->pluck('name'))),

            // sales credits
            'sales_credits' => SalesCreditResource::collection($this->whenLoaded('salesCredits', function () {
                return $this->salesCredits;
            })),
            'resend' => $this->inclusive('resend', $this->whenLoaded('resendLinks', function () {
                return $this->sales_resend;
            })),
            'parent_resend' => $this->inclusive('parent_resend', $this->whenLoaded('resendParentsLinks', function () {
                return $this->parent_sales_resend;
            })),

            'warehouses' => $this->inclusive('warehouses', $this->whenLoaded('salesOrderLines', function () {
                return $this->salesOrderLines->filter->warehouse->unique->warehouse->map(function ($warehouse) {
                    return [
                        'id' => $warehouse->warehouse->id,
                        'name' => $warehouse->warehouse->name,
                        'supplier_id' => $warehouse->warehouse->supplier_id,
                        'supplier_name' => $warehouse->warehouse->supplier ? $warehouse->warehouse->supplier->name : null,
                    ];
                })->toArray();
            })),

            'mapping_needed' => $this->inclusive('mapping_needed', $this->whenLoaded('salesOrderLines', function () {
                return (bool) $this->salesOrderLines->where('product_id', null)->where('is_product', '=', 1)->count();
            })),

            'can_fulfill' => $this->inclusive('fulfillable', $this->whenLoaded('salesOrderLines', function () {
                return $this->whenLoaded('salesOrderFulfillments', function () {
                    return $this->whenLoaded('salesOrderLines.salesOrderFulfillmentLines', function () {
                        return $this->whenLoaded('saleOrderLines.backorderQueue', function () {
                            return $this->is_fulfillable;
                        });
                    });
                });
            })),
            'exact_discount_total' => $this->inclusive('exact_discount_total', $this->exact_discount_total),
            'is_tax_included' => $this->is_tax_included,
            'tax_total' => $this->tax_total,
            'calculated_tax_total' => $this->inclusive('calculated_tax_total', $this->calculated_tax_total),
            'exact_tax_total' => $this->inclusive('exact_tax_total', $this->exact_tax_total),
            'tax_rate_id' => $this->inclusive('tax_rate_id', $this->tax_rate_id),
            'tax_rate' => $this->inclusive('tax_rate', $this->taxRate),
            'payments' => $this->inclusive('payments', $this->whenLoaded('payments', function () {
                return PaymentResource::collection($this->payments);
            })),
            /*
             * TODO: Commented out 3 lines in the next code block because they were returning order_lines
             */
            'customer_recent_sales_orders' => $this->inclusive('payments', $this->whenLoaded('customer.salesOrders', function () {
                return $this->customer->salesOrders()
                    ->select('sales_orders.id', 'sales_order_number', 'order_date')
                    //->selectRaw('SUM(sales_order_lines.quantity * sales_order_lines.amount) as total')
                    //->join('sales_order_lines', 'sales_order_lines.sales_order_id', '=', 'sales_orders.id')
                    ->where('sales_orders.id', '!=', $this->id)
                    ->orderBy('sales_orders.created_at', 'desc')
                    //->groupBy(['sales_orders.id', 'sales_order_number', 'order_date'])
                    ->limit(3)
                    ->get();
            })),
            'notes' => $this->inclusive('notes', $this->whenLoaded('notes', function () {
                return $this->notes()
                    ->select('notes.id', 'notes.note', 'notes.created_at')
                    ->selectRaw('IFNULL(users.name, "System") as user_name')
                    ->leftJoin('users', 'users.id', '=', 'notes.user_id')
                    ->orderByDesc('created_at')
                    ->get();
            })),

            $this->mergeWhen($this->relationLoaded('backorderedLines'), function () {
                return [
                    'earliest_covered_ship_by_date' => $this->backorderedLines->min(function($orderLine) {
                        return $orderLine->backorderQueue->backorderQueueCoverages->min(function($backorder) {
                            return $backorder->purchaseOrderLine->estimated_delivery_date ?? $backorder->purchaseOrderLine->purchaseOrder->estimated_delivery_date;
                        });
                    }),
                    'latest_covered_ship_by_date' => $this->latest_covered_ship_by_date,
                    'has_backorder_lines' => (bool) $this->backorderedLines->count(),
                ];
            })
        ];
    }
}
