<?php

namespace App\Http\Resources;

use App\Models\Amazon\FBARealTimeInventory;
use App\Models\Product;
use App\Models\ProductInventory;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Repositories\ProductRepository;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;

/**
 * Class SalesOrderLineInventoryResource.
 *
 *
 * @mixin Product
 */
class SalesOrderLineInventoryResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        /** @var Collection|ProductInventory[] $productInventory */
        $productInventory = $this->productInventory;

        /** @var ProductInventory $totalInventory */
        $totalInventory = $productInventory->firstWhere('warehouse_id', ProductInventory::$idForTotalWarehouses);
        $inventory = [
            'total' => [
                'inbound' => app(ProductRepository::class)->getInboundQuantityForProduct($this->getModel()),
                'stock' => $totalInventory->inventory_total ?? 0,
                'reserved' => $totalInventory->inventory_reserved ?? 0,
                'available' => $totalInventory->inventory_available ?? 0,
                'in_transit' => $totalInventory->inventory_in_transit ?? 0,
            ],
        ];

        $inventory['warehouses'] = [];

        // 1- if the supplier is sourcing and dropship is enabled
        // 2- if not supplier warehouses we will add all of them (not necessarily have movements)
        $productWarehouses = Warehouse::with([])
            ->whereNull('supplier_id')
            ->orWhereIn('id', $this->getSupplierDropshipWarehouses())
            ->archived(0)
            ->get(['id', 'name', 'supplier_id', 'type', 'integration_instance_id']);

        /** @var Warehouse $warehouse */
        foreach ($productWarehouses as $warehouse) {
            /** @var ProductInventory $warehouseInventory */
            $warehouseInventory = $productInventory->firstWhere('warehouse_id', $warehouse->id);
            $supplierInventory = $this->suppliersInventory->firstWhere('warehouse_id', $warehouse->id);

            $inventory['warehouses'][$warehouse->name] = [
                'warehouse_id' => $warehouse->id,
                'warehouse_name' => $warehouse->name,
                'warehouse_type' => $warehouse->type,
                'supplier' => $warehouse->isSupplierWarehouse() ? [
                    'name' => $warehouse->supplier->name,
                    'id' => $warehouse->supplier->id,
                ] : null,
            ];
            $inventory['warehouses'][$warehouse->name] += [
                'inbound' => app(ProductRepository::class)->getInboundQuantityForProduct($this->getModel(), $warehouse),
                'stock' => $warehouse->isSupplierWarehouse() ? ($supplierInventory->stock ?? 0) : ($warehouseInventory->inventory_total ?? 0),
                'reserved' => $warehouseInventory->inventory_reserved ?? 0,
                'available' => $warehouse->isSupplierWarehouse() ? ($supplierInventory->stock ?? 0) : ($warehouseInventory->inventory_available ?? 0),
                'in_transit' => $warehouseInventory->inventory_in_transit ?? 0,
            ];
        }

        return $inventory;
    }

    private function getSupplierDropshipWarehouses()
    {
        $suppliersDropshipWarehousesId = [];
        $this->suppliers->map(function (Supplier $supplier) use (&$suppliersDropshipWarehousesId) {
            $dropshipWarehouses = $supplier->warehouses->where('dropship_enabled', true);
            if ($dropshipWarehouses->isNotEmpty()) {
                $suppliersDropshipWarehousesId = array_merge($suppliersDropshipWarehousesId, $dropshipWarehouses->pluck('id')->toArray());
            }
        });

        return $suppliersDropshipWarehousesId;
    }
}
