<?php

namespace App\Http\Resources;

use App\DataTable\DataTableResource;
use App\Models\SalesCredit;
use Illuminate\Http\Request;

/**
 * Class SalesCreditResource.
 *
 *
 * @mixin SalesCredit
 */
class SalesCreditResource extends DataTableResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        $payments = $this->whenLoaded('payments', function () {
            $payments = collect($this->payments);

            return $payments->merge($this->allocations);
        });

        return [
            //-----------------------------------------------------
            // Basic Info
            //-----------------------------------------------------

            'id' => $this->id,
            'sales_credit_number' => $this->inclusive('sales_credit_number', $this->sales_credit_number),
            'store' => $this->inclusive('store', $this->whenLoaded('store', function () {
                return $this->store->only('id', 'name', 'email');
            })),
            'credit_date' => $this->inclusive('credit_date', $this->credit_date),
            //      'payment_date'       => $this->inclusive('payment_date', $this->fully_paid_at ?: function () {
            //        return $this->payments->first()->payment_date ?? null;
            //      }),
            'integration' => $this->inclusive('integration_name', $this->when(! is_null($this->salesOrder), function () {
                return $this->salesOrder->salesChannel->integration->name ?? null;
            })),
            'is_for_overpayment' => $this->inclusive('is_for_overpayment', $this->is_for_overpayment),
            'created_at' => $this->inclusive('created_at', $this->created_at),
            'updated_at' => $this->inclusive('updated_at', $this->updated_at),
            'archived_at' => $this->inclusive('archived_at', $this->archived_at),

            //-----------------------------------------------------
            // Returns
            //-----------------------------------------------------

            'to_warehouse' => $this->inclusive('to_warehouse', $this->whenLoaded('toWarehouse', function () {
                return $this->toWarehouse->only('id', 'name');
            })),
            'returns_receipt_date' => $this->inclusive('returns_receipt_date', $this->fully_returned_at ?: function () {
                return $this->salesCreditReturns->first()->received_at ?? null;
            }),
            'returns_receipt_quantity' => $this->inclusive('returns_receipt_quantity', $this->whenLoaded('salesCreditLines', function () {
                return $this->salesCreditLines->where('is_product', true)->sum('received_quantity');
            })),
            'returns_receipt_items' => $this->inclusive('returns_info', ReturnReceiptItemResource::collection($this->whenLoaded('salesCreditLines'))),

            //-----------------------------------------------------
            // Customer
            //-----------------------------------------------------

            $this->mergeWhen($this->whenLoaded('customer', true, false) && $this->customer->relationLoaded('address'), function () {
                return SalesCustomerAddressResource::make($this->customer);
            }),

            //-----------------------------------------------------
            // Items
            //-----------------------------------------------------

            'item_info' => $this->inclusive('item_info', SalesCreditLineResource::collection($this->whenLoaded('salesCreditLines'))),

            //-----------------------------------------------------
            // Statuses
            //-----------------------------------------------------

            'statuses' => [
                'credit_status' => $this->inclusive('credit_status', $this->credit_status),
                //        'payment_status' => $this->inclusive('payment_status', $this->payment_status),
                'return_status' => $this->inclusive('return_status', $this->return_status),
            ],

            //-----------------------------------------------------
            // Totals
            //-----------------------------------------------------

            'additional_credit' => $this->inclusive('additional_credit', $this->additional_credit),
            'tax_credit' => $this->inclusive('tax_credit', $this->tax_credit),
            'product_total' => $this->inclusive('product_total', $this->product_total),
            'total' => $this->inclusive('total', $this->total_credit),
            'currency_code' => $this->inclusive('currency_code', $this->currency->code ?? null),

            //-----------------------------------------------------
            // Other
            //-----------------------------------------------------

            'tags' => $this->inclusive('tags', $this->tags->pluck('name')),

            'sales_order' => $this->whenLoaded('salesOrder', function () {
                return $this->salesOrder ? $this->salesOrder->only('id', 'sales_order_number') : null;
            }),

            'exchange_orders' => $this->exchange_orders ? SalesOrderResource::collection($this->exchange_orders) : [],

            'payments' => PaymentResource::collection($payments),

            'has_products' => $this->inclusive('has_products', $this->has_products),
            'tax_lines' => $this->inclusive('tax_lines', $this->tax_lines),
            'tax_rate' => $this->inclusive('tax_rate', $this->taxRate),
            'calculated_tax_total' => $this->inclusive('calculated_tax_total', $this->calculated_tax_total),

            'sales_credit_note' => $this->inclusive('sales_credit_note', $this->sales_credit_note),

            'is_tax_included' => $this->inclusive('is_tax_included', $this->is_tax_included),
            'tax_rate_id' => $this->inclusive('tax_rate_id', $this->tax_rate_id),
        ];
    }
}
