<?php

namespace App\Http\Resources;

use App\DataTable\DataTableResource;
use App\Models\Product;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentReceiptLine;
use Illuminate\Http\Request;

/**
 * Class PurchaseOrderLineResource.
 *
 *
 * @mixin PurchaseOrderLine
 */
class PurchaseOrderLineResource extends DataTableResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        /** @var Product $product */
        $product = $this->resource->relationLoaded('product') ? $this->product : null;

        return [
            'id' => $this->id,
            'quantity' => $this->quantity,
            'sku' => [
                'id' => $this->product_id,
                'sku' => $product->sku ?? null
            ],
            'barcode' => $product?->barcode,
            'is_variation' => $this->product ? $this->product->is_variation : false,
            'matrix_product' => $this->when($this->product !== null && $this->product->is_variation, function () {
                return [
                    'id' => $this->product->parent->id,
                    'sku' => $this->product->parent->sku,
                ];
            }),
            'img_url' => $product?->primaryImage?->url,
            'supplier_sku' => $this->supplierProduct?->supplier_sku ?? ($request->input('to_export') ? ($product->sku ?? null) : null),
            'price' => $this->amount,
            'name' => $this->description,
            'nominal_code_id' => $this->nominal_code_id,
            'nominal_code' => $this->nominalCode->name ?? null,
            'received' => $this->received_quantity,
            'invoiced' => $this->when($request->input('to_export'), function () {
                return $this->quantity - $this->uninvoiced_quantity;
            }),
            'uninvoiced_quantity' => $this->uninvoiced_quantity,
            'eta' => $this->estimated_delivery_date,
            'tax_rate' => $this->tax_rate,
            'tax_rate_id' => $this->tax_rate_id,
            'tax_allocation' => $this->tax_allocation,
            'discount' => $this->discount,
            'discount_rate' => $this->discount_rate,
            'discount_amount' => $this->discount_amount,
            'discount_amount_extended' => $this->discount_amount_extended,
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,

            $this->mergeWhen($product && $this->relationLoaded('product.totalInventory'), function () use ($product) {
                return [
                    'stock' => $this->purchaseOrder->destinationWarehouse
                            ? $product->getWarehouseStock($this->purchaseOrder->destinationWarehouse)
                            : $product->total_stock,
                ];
            }),

            $this->mergeWhen($product && $product->relationLoaded('suppliersInventory'), function () use ($product) {
                return [
                    'supplier_stock' => $product->suppliersInventory ? $product->suppliersInventory->firstWhere('supplier_id', $this->purchaseOrder->supplier_id) : 0,
                ];
            }),

            $this->mergeWhen($product && $product->relationLoaded('productInventory'), function () use ($product) {
                return ['inventory' => SalesOrderLineInventoryResource::make($product)];
            }),
            'backorder_coverage' => $this->when(isset($this->additional(['loadedRelations'])['coveredBackorderQueues']), function () {
                return BackorderQueueCoverageResource::collection($this->coveredBackorderQueues);
            }),
            'backorder_releases' => $this->when(isset($this->additional(['loadedRelations'])['purchaseOrderShipmentReceiptLines']), function () {
                return BackorderQueueReleaseResource::collection($this->purchaseOrderShipmentReceiptLines->map(fn (PurchaseOrderShipmentReceiptLine $receiptLine) => $receiptLine->backorderQueueReleases)->collapse());
            }),
        ];
    }
}
