<?php

namespace App\Http\Resources;

use App\Models\Product;
use App\Models\ProductInventory;
use App\Models\Warehouse;
use App\Repositories\ProductRepository;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Modules\Amazon\Repositories\AmazonProductRepository;

/**
 * Class POBuilderSalesOrderLineInventoryResource.
 *
 * @mixin Product
 */
class POBuilderSalesOrderLineInventoryResource extends JsonResource
{
    /**
     * @var int The destination warehouse of the PO builder
     */
    public static $warehouseId = null;

    private AmazonProductRepository $amazonProductRepository;

    public function __construct($resource)
    {
        parent::__construct($resource);
        $this->amazonProductRepository = app(AmazonProductRepository::class);
    }

    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        /** @var Collection|ProductInventory[] $productInventory */
        $productInventory = $this->productInventory();

        $inventory['warehouses'] = [];

        if (self::$warehouseId){
            $warehouse = Warehouse::with([])->findOrFail(self::$warehouseId);
            $warehouseInventory = $productInventory->where('warehouse_id', $warehouse->id)->first();

            $inventory['warehouses'][$warehouse->name] = [
                'inbound' => app(ProductRepository::class)->getInboundQuantityForProduct($this->getModel(), $warehouse),
                'stock' => $warehouseInventory->inventory_total ?? 0,
                'reserved' => $warehouseInventory->inventory_reserved ?? 0,
                'available' => $warehouseInventory->inventory_available ?? 0,
                'in_transit' => $warehouseInventory->inventory_in_transit ?? 0,
                'warehouse_id' => $warehouse->id,
                'warehouse_name' => $warehouse->name,
            ];
        } else {

            $fbaWarehouses = Warehouse::query()->where('type', Warehouse::TYPE_AMAZON_FBA)->pluck('id')->toArray();

            $productInventory = $productInventory
                ->where('warehouse_id', '!=', 0)
                ->whereNotIn('warehouse_id', $fbaWarehouses);

            $warehousesFBARealTimeData = $this->amazonProductRepository->getFBAReportInventoryForIntegrationInstances($this->id);

            $nonFBAInbound = app(ProductRepository::class)
                ->getInboundQuantityForProduct(
                    product: $this->getModel(),
                    excludeWarehouseTypes: [Warehouse::TYPE_AMAZON_FBA, Warehouse::TYPE_SUPPLIER]
                );

            $reserved = $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'reserved');

            $inventory['warehouses']['all'] = [
                'inbound' => $nonFBAInbound + $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'inbound'),
                'stock' => $productInventory->sum('inventory_total') + $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData),
                'reserved' => $productInventory->sum('inventory_reserved') + $reserved,
                'available' => $productInventory->sum('inventory_available') + $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'available') + $reserved,
                'in_transit' => $productInventory->sum('inventory_in_transit'),
                'warehouse_id' => 0,
                'warehouse_name' => 'All Warehouses',
            ];
        }

        return $inventory;
    }
}
