<?php

namespace App\Http\Resources;

use App\DataTable\DataTableResource;
use App\Models\SalesOrderFulfillment;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\MergeValue;
use Illuminate\Http\Resources\MissingValue;

/**
 * @mixin SalesOrderFulfillment
 */
class FulfillmentResource extends DataTableResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        return [
            //-----------------------------------------------------
            // Basic Info
            //-----------------------------------------------------

            'fulfillment_number' => $this->inclusive('fulfillment_number', $this->whenLoaded('salesOrder', function () {
                return $this->fulfillment_number;
            })),
            'fulfillment_sequence' => $this->inclusive('fulfillment_number', $this->fulfillment_sequence),
            'fulfillment_type' => $this->inclusive('fulfillment_type', $this->fulfillment_type),
            'warehouse' => $this->inclusive('warehouse', $this->whenLoaded('warehouse', function () {
                return $this->warehouse->only(['id', 'name']);
            })),
            'requested_shipping_method_id' => $this->requested_shipping_method_id,
            'requested_shipping_method' => $this->inclusive('requested_shipping_method', $this->when($this->relationLoaded('requestedShippingMethod'), function () {
                return $this->requestedShippingMethod ? $this->requestedShippingMethod->full_name : $this->requested_shipping_method;
            })),
            'fulfilled_shipping_method_id' => $this->fulfilled_shipping_method_id,
            'fulfillment_shipping_method' => $this->inclusive('fulfillment_shipping_method', $this->when($this->relationLoaded('fulfilledShippingMethod'), function () {
                return $this->fulfilledShippingMethod ? $this->fulfilledShippingMethod->full_name : $this->fulfilled_shipping_method;
            })),
            'tracking_number' => $this->inclusive('tracking_number', $this->tracking_number),
            'tracking_link' => $this->inclusive('tracking_number', $this->tracking_link),
            'fulfilled_at' => $this->inclusive('fulfilled_at', $this->fulfilled_at),
            'packing_slip_printed_at' => $this->inclusive('packing_slip_printed_at', $this->packing_slip_printed_at),
            'cost' => $this->inclusive('cost', $this->cost),
            'created_at' => $this->inclusive('created_at', $this->created_at),
            'updated_at' => $this->inclusive('updated_at', $this->updated_at),
            'archived_at' => $this->inclusive('archived_at', $this->archived_at),
            'id' => $this->id,
            'sales_order' => $this->whenLoaded('salesOrder', function () {
                return ['id' => $this->sales_order_id, 'sales_order_number' => $this->salesOrder->sales_order_number];
            }),

            //-----------------------------------------------------
            // Sales Order
            //-----------------------------------------------------

            'customer_reference' => $this->inclusive('customer_reference', $this->whenLoaded('salesOrder', function () {
                return $this->salesOrder->sales_order_number;
            })),
            'store' => $this->inclusive('store', $this->whenLoaded('salesOrder', function () {
                return $this->salesOrder->active_store ? $this->salesOrder->active_store->name : null;
            })),
            'sales_channel_type_name' => $this->inclusive('sales_channel_type_name', $this->whenLoaded('salesOrder', function () {
                return $this->salesOrder->salesChannel->integration->name;
            })),
            'sales_order_shipping_method' => $this->inclusive('sales_order_shipping_method', $this->whenLoaded('salesOrder', function () {
                return $this->salesOrder->shippingMethod ? $this->salesOrder->shippingMethod->full_name : $this->salesOrder->requested_shipping_method;
            })),

            //-----------------------------------------------------
            // Customer
            //-----------------------------------------------------

            $this->whenLoaded('salesOrder', function () {
                if ($this->salesOrder->relationLoaded('customer')) {
                    return new MergeValue(SalesCustomerAddressResource::make($this->salesOrder->customer));
                }

                return new MissingValue();
            }, new MissingValue()),

            //-----------------------------------------------------
            // Items
            //-----------------------------------------------------

            'item_info' => $this->inclusive('item_info', FulfillmentLineResource::collection($this->whenLoaded('salesOrderFulfillmentLines'))),

            //-----------------------------------------------------
            // Statuses
            //-----------------------------------------------------

            'status' => $this->inclusive('status', $this->status),
        ];
    }
}
