<?php

namespace App\Http\Requests;

use App\Models\InboundShipmentRelation;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Validator;
use Carbon\Carbon;
use Illuminate\Foundation\Http\FormRequest;

class StoreWarehouseTransfer extends FormRequest
{
    use InstanceFromCustom;

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'transfer_date' => 'nullable|required_with:eta|date',
            'eta' => 'nullable|date',
            'from_warehouse_id' => 'required|exists:warehouses,id',
            'from_warehouse_location_id' => 'nullable|exists:warehouse_locations,id',
            'to_warehouse_id' => 'required|exists:warehouses,id|different:from_warehouse_id',
            'products' => 'nullable|array|min:1',
            'products.*.id' => 'sometimes|exists:products,id',
            'products.*.quantity' => 'required|numeric|min:1',
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'shipment' => 'nullable|array',
            'shipment.cost' => 'nullable|numeric|min:0|lt:100000',
            'shipment.tracking_number' => 'nullable',
            'shipment.shipment_date' => 'required_with:shipment|date',
            'shipment_id' => 'nullable',
            'order_id' => 'nullable',
        ];

        if ($this->isMethod('put')) {
            $rules['transfer_date'] = 'sometimes|'.$rules['transfer_date'];
            $rules['from_warehouse_id'] = 'sometimes|'.$rules['from_warehouse_id'];
            $rules['to_warehouse_id'] = 'sometimes|'.$rules['to_warehouse_id'];
            $rules['products'] = 'sometimes|'.$rules['products'];
            $rules['shipping_method_id'] = 'sometimes|'.$rules['shipping_method_id'];

            // We don't take shipment in update
            unset($rules['shipment']);
        }

        return $rules;
    }

    public function withValidator(Validator $validator)
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // From warehouse must have a default location
                if (isset($attributes['from_location_id'])) {
                    $defaultFromWarehouseLocation = Warehouse::with([])->findOrFail($attributes['from_warehouse_id'])->defaultLocation;
                    if (! $defaultFromWarehouseLocation) {
                        $validator->addFailure('from warehouse location', 'NoDefaultFromWarehouseLocation');
                    }
                }

                // To warehouse must have a default location
                if (isset($attributes['to_location_id'])) {
                    $defaultToWarehouseLocation = Warehouse::with([])->findOrFail($attributes['to_warehouse_id'])->defaultLocation;
                    if (! $defaultToWarehouseLocation) {
                        $validator->addFailure('to warehouse location', 'NoDefaultToWarehouseLocation');
                    }
                }

                // Eta must not be before transfer date
                if (! empty($attributes['eta'])) {
                    $eta = new Carbon($attributes['eta']);
                    if (empty($attributes['transfer_date'])) {
                        $attributes['transfer_date'] = WarehouseTransfer::with([])->findOrFail($this->route('transfer'))->transfer_date;
                    }
                    $transferDate = new Carbon($attributes['transfer_date']);
                    if ($eta->isBefore($transferDate)) {
                        $validator->addFailure('eta', 'EtaBeforeTransferDate');
                    }
                }
                if (! empty($attributes['to_warehouse_id'])) {
                    $toWarehouse = Warehouse::with([])->findOrFail($attributes['to_warehouse_id']);
                    if ($toWarehouse->type == Warehouse::TYPE_AMAZON_FBA) {
                        if (empty($attributes['shipment_id'])) {
                            $validator->addFailure('shipment_id', 'Required');
                        }
                        if (isset($attributes['shipment_id']) && InboundShipmentRelation::with([])->where('shipment_id', $attributes['shipment_id'])->exists()) {
                            $validator->addFailure('shipment_id', 'ShipmentUsed');
                        }
                    }
                }
                if ($this->isMethod('PUT')) {
                    $warehouse = $this->route('transfer') instanceof WarehouseTransfer ? $this->route('transfer') : WarehouseTransfer::with([])->findOrFail($this->route('transfer'));
                    if ($warehouse->toWarehouse->type == Warehouse::TYPE_AMAZON_FBA && $warehouse->transfer_status != WarehouseTransfer::TRANSFER_STATUS_DRAFT) {
                        $validator->addFailure('to warehouse location', 'CannotEditFbaWarehouse');
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'no_default_from_warehouse_location' => 'From warehouse must have a default location.',
            'to_warehouse_id.required' => 'The Destination warehouse is required',
            'no_default_to_warehouse_location' => 'To warehouse must have a default location.',
            'products_min' => 'There must be at least 1 product in the warehouse transfer.',
            'eta_before_transfer_date' => 'ETA cannot be before transfer date.',
            'cannot_edit_fba_warehouse' => 'Cannot edit when destination is an fba warehouse.',
            'shipment_used' => 'Shipment Id already used.',
        ];
    }
}
