<?php

namespace App\Http\Requests;

use App\Models\Warehouse;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

/**
 * Class StoreWarehouse.
 */
class StoreWarehouse extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'name' => 'required|max:255',
            'address_name' => 'max:255',
            'type' => 'required|in:'.implode(',', Warehouse::TYPES),
            'company' => 'nullable|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|max:255',
            'fax' => 'nullable|max:255',
            'address1' => 'nullable|max:255',
            'address2' => 'nullable|max:255',
            'address3' => 'nullable|max:255',
            'city' => 'nullable|max:255',
            'province' => 'nullable|max:255',
            'province_code' => 'nullable|max:255',
            'zip' => 'nullable|max:255',
            'country' => 'nullable|max:255',
            'country_code' => 'nullable|max:255',
            'supplier_id' => 'sometimes|exists:suppliers,id',
            'order_fulfillment' => 'nullable|string|max:255',
            'dropship_enabled' => 'boolean',
            'direct_returns' => 'boolean',
            'auto_routing_enabled' => 'boolean',
            'customer_returns' => 'boolean',
            'default_location' => 'nullable|array',
            'default_location.aisle' => 'nullable|max:255',
            'default_location.bay' => 'nullable|max:255',
            'default_location.shelf' => 'nullable|max:255',
            'default_location.bin' => 'nullable|max:255',
            'is_default' => 'boolean',
            'nominal_code_id' => 'nullable|exists:nominal_codes,id',
        ];

        if ($this->isMethod('put')) {
            $rules['name'] = 'sometimes|'.$rules['name'];
            $rules['type'] = 'sometimes|'.$rules['type'];
        }

        if ($this->route('supplier')) {
            $rules['type'] = 'sometimes|'.$rules['type'];
        }

        return $rules;
    }

    public function withValidator(Validator $validator)
    {
        if ($validator->passes()) {
            // Check for unique warehouse name
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                if ($this->isMethod('post')) {
                    $this->checkUniqueNameForPost($attributes['name'], $validator);
                } elseif (isset($attributes['name'])) {
                    $this->checkUniqueNameForUpdate($attributes['name'], $validator);
                }
            });
        }
    }

    private function checkUniqueNameForUpdate($name, Validator $validator)
    {
        // We get the warehouse
        $warehouse = Warehouse::with([])->findOrFail(e($this->route('warehouse')));

        if ($warehouse->isSupplierWarehouse() && $this->supplierHasWarehouseName($warehouse->supplier_id, $name, $warehouse)) {
            // Duplicate supplier warehouse name
            $validator->addFailure('name', 'MustBeUniqueForSupplier');
        } elseif (! $warehouse->isSupplierWarehouse() && $this->warehouseNameExists($name)) {
            // Updating non-supplier warehouse
            $validator->addFailure('name', 'MustBeUnique');
        }
    }

    private function checkUniqueNameForPost($name, Validator $validator)
    {
        if ($supplier = $this->route('supplier')) {
            // Supplier warehouse
            if (($supplierId = $supplier->id) && $this->supplierHasWarehouseName($supplierId, $name)) {
                // Duplicate supplier warehouse name
                $validator->addFailure('name', 'MustBeUniqueForSupplier');
            }
        } elseif ($this->warehouseNameExists($name)) {
            // Duplicate warehouse name
            $validator->addFailure('name', 'MustBeUnique');
        }
    }

    public function messages(): array
    {
        return [
            'must_be_unique_for_supplier' => 'Warehouse name should be unique for a given supplier.',
            'must_be_unique' => 'Warehouse name should be unique.',
        ];
    }

    private function supplierHasWarehouseName($supplierId, $name, $warehouse = null): bool
    {
        $query = Warehouse::with([])->where('supplier_id', $supplierId)
            ->where('name', $name);

        if ($warehouse) {
            $query = $query->where('id', '!=', $warehouse->id);
        }

        return $query->first() != null;
    }

    private function warehouseNameExists($name): bool
    {
        return Warehouse::with([])->where('name', $name)->first() != null;
    }
}
