<?php

namespace App\Http\Requests;

use App\Models\Supplier;
use App\Validator;
use Egulias\EmailValidator\EmailValidator;
use Egulias\EmailValidator\Validation\RFCValidation;
use Illuminate\Foundation\Http\FormRequest;

class StoreSupplier extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'name' => 'required|max:255|unique:suppliers,name',
            'company_name' => 'nullable|max:255',
            'primary_contact_name' => 'nullable|max:255',
            'email' => 'nullable|max:255',
            'purchase_order_email' => 'nullable|max:255',
            'website' => 'nullable|max:255',
            'leadtime' => 'nullable|integer',
            'minimum_order_quantity' => 'nullable|numeric',
            'minimum_purchase_order' => 'nullable|numeric|lt:100000',
            'pricing' => 'nullable|array',
            'pricing.*.id' => 'required|exists:supplier_pricing_tiers,id',
            'address' => 'nullable|array',
            'address.name' => 'nullable|max:255',
            'address.company' => 'nullable|max:255',
            'address.email' => 'nullable|email',
            'address.phone' => 'nullable:max:255',
            'address.fax' => 'nullable:max:255',
            'address.address1' => 'nullable|max:255',
            'address.address2' => 'nullable|max:255',
            'address.address3' => 'nullable|max:255',
            'address.city' => 'nullable|max:255',
            'address.province' => 'nullable|max:255',
            'address.province_code' => 'nullable|max:255',
            'address.zip' => 'nullable|max:255',
            'address.country' => 'nullable|max:255',
            'address.country_code' => 'nullable|max:2',
            'address.label' => 'required_with:address|max:255',
            'default_pricing_tier_id' => 'nullable|exists:supplier_pricing_tiers,id',
            'default_warehouse_id' => 'nullable|exists:warehouses,id',
            'default_store_id' => 'nullable|exists:stores,id',
            'po_processing_method' => 'nullable|in:'.implode(',', Supplier::PO_DELIVERY_METHODS),
            'po_batch_schedule' => 'nullable|array',
            'po_batch_schedule.*.time' => 'required_with:po_batch_schedule',
            'po_batch_schedule.*.days' => 'required_with:po_batch_schedule|array|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'default_shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'auto_fulfill_dropship' => 'nullable|boolean',
            'auto_submit_dropship_po' => 'nullable|boolean',
            'auto_generate_backorder_po' => 'nullable|boolean',
            'auto_submit_backorder_po' => 'nullable|boolean',
            'auto_split_backorder_po_by_brand' => 'nullable|boolean',
            'auto_receive_backorder_po' => 'nullable|boolean',
            'backorder_po_schedule' => 'nullable|array',
            'backorder_po_schedule.*.time' => 'required_with:backorder_po_schedule',
            'backorder_po_schedule.*.days' => 'required_with:backorder_po_schedule|array|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'default_stock_level' => 'in:'.implode(',', Supplier::STOCK_LEVELS),
            'default_tax_rate_id' => 'nullable|exists:tax_rates,id',
            'timezone' => 'sometimes|timezone',
        ];

        if ($this->getMethod() == 'PUT') {
            $rules['name'] = 'sometimes|required|max:255|unique:suppliers,name,'.$this->route('supplier');
            $rules['purchase_order_format'] = 'nullable';
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        $validator->after(function (Validator $validator) {
            $attributes = $validator->attributes();
            // check email
            if (! empty($attributes['email'])) {
                $this->validateEmails($validator, 'email');
            }
            // check purchase_order_email
            if (! empty($attributes['purchase_order_email'])) {
                $this->validateEmails($validator, 'purchase_order_email');
            }
        });
    }

    private function validateEmails(Validator $validator, string $attribute)
    {
        $attributes = $validator->attributes();

        foreach (explode(',', $attributes[$attribute]) as $email) {
            if (! (new EmailValidator)->isValid($email, new RFCValidation())) {
                $validator->addFailure($attribute, 'Email');
            }
        }
    }
}
