<?php

namespace App\Http\Requests;

use App\Models\ShippingProvider;
use App\Models\Warehouse;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Validator;

class StoreShippingProvider extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'name' => 'required|max:255|unique:shipping_providers,name,'.$this->route('shipping_provider'),
            'type' => 'required|in:'.implode(',', ShippingProvider::TYPES),
            'warehouse_ids' => 'required',
        ];

        if ($this->getMethod() === 'PUT') {
            $rules['name'] = 'sometimes|'.$rules['name'];
            $rules['type'] = 'sometimes|'.$rules['type'];
            $rules['warehouse_ids'] = 'sometimes|'.$rules['warehouse_ids'];
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (\App\Validator $validator) {
                $attributes = $validator->attributes();

                if ($this->getMethod() === 'PUT' && ! isset($attributes['warehouse_ids'])) {
                    return;
                }

                if (is_array($attributes['warehouse_ids']) and
             Warehouse::with([])->whereIn('id', $attributes['warehouse_ids'])->count() != count($attributes['warehouse_ids'])) {
                    $validator->addFailure('warehouse_ids', 'Exists');
                }

                if (is_string($attributes['warehouse_ids']) and $attributes['warehouse_ids'] != 'ALL') {
                    $validator->addFailure('warehouse_ids', 'MustBeArrayOrAll');
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'exists' => 'The warehouse ids have invalid warehouses ids.',
            'must_be_array_or_all' => 'The warehouse ids must be array or ALL',
        ];
    }
}
