<?php

namespace App\Http\Requests;

use App\Models\SalesCredit;
use App\Models\SalesOrderLine;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class StoreSalesCredit extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'sales_order_id' => 'required_without:customer_id|exists:sales_orders,id',
            'sales_credit_number' => 'nullable|max:255',
            'to_warehouse_id' => 'nullable|exists:warehouses,id',
            'sales_credit_note' => 'nullable',
            'credit_date' => 'nullable|date',
            'store_id' => 'nullable|exists:stores,id',
            'customer_id' => 'required_without:sales_order_id',
            'currency_id' => 'nullable|exists:currencies,id',
            'currency_code' => 'nullable|exists:currencies,code',
            'tax_lines' => 'nullable|array',
            'tax_rate_id' => 'nullable|numeric|exists:tax_rates,id',
            'is_tax_included' => 'nullable|boolean',

            // lines

            'sales_credit_lines' => 'required|array|min:1',
            'sales_credit_lines.*.sales_order_line_id' => 'required_without:sales_credit_lines.*.description|exists:sales_order_lines,id',
            'sales_credit_lines.*.description' => 'required_without:sales_credit_lines.*.sales_order_line_id',
            'sales_credit_lines.*.quantity' => 'required|numeric|min:1',
            'sales_credit_lines.*.tax_allocation' => 'nullable|numeric|lt:100000',
            'sales_credit_lines.*.amount' => 'nullable|numeric|lt:100000',
            'sales_credit_lines.*.tax_rate_id' => 'nullable|numeric|exists:tax_rates,id',
            'sales_credit_lines.*.is_product' => 'nullable|boolean',
            'sales_credit_lines.*.nominal_code_id' => 'nullable|exists:nominal_codes,id',
        ];

        if ($this->isMethod(self::METHOD_PUT)) {
            unset($rules['sales_order_id']);
            //      $rules['to_warehouse_id'] = 'sometimes|' . $rules['to_warehouse_id'];

            $rules['customer_id'] = 'sometimes|'.$rules['customer_id'];

            // lines
            $rules['sales_credit_lines'] = 'sometimes|'.$rules['sales_credit_lines'];
            $rules['sales_credit_lines.*.id'] = 'nullable|exists:sales_credit_lines,id';
            $rules['sales_credit_lines.*.quantity'] = 'required_without:sales_credit_lines.*.id|numeric|min:1';
            $rules['sales_credit_lines.*.sales_order_line_id'] = 'nullable|exists:sales_order_lines,id';
            $rules['sales_credit_lines.*.nominal_code_id'] = 'nullable|exists:nominal_codes,id';
            $rules['sales_credit_lines.*.description'] = 'required_without:sales_credit_lines.*.id|max:255';
            $rules['sales_credit_lines.*.amount'] = 'required_without:sales_credit_lines.*.id|numeric|lt:100000';
            $rules['sales_credit_lines.*.discount'] = 'nullable|numeric';
            $rules['sales_credit_lines.*.tax_rate_id'] = 'nullable|numeric|exists:tax_rates,id';
            $rules['sales_credit_lines.*.is_product'] = 'nullable|boolean';
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                $salesOrderId = $attributes['sales_order_id'] ?? null;

                // check sales order lines belong to the sales order and unreturned
                if (! empty($attributes['sales_credit_lines'])) {
                    $salesOrderLines = SalesOrderLine::with(['salesCreditLines', 'salesOrderFulfillmentLines'])
                        ->whereIn('id', array_column($attributes['sales_credit_lines'], 'sales_order_line_id'))
                        ->get();

                    if (empty($salesOrderId) && $this->isMethod(self::METHOD_PUT)) {
                        /** @var SalesCredit $salesCredit */
                        $salesCredit = $this->route('sales_credit');
                        $salesOrderId = $salesCredit->sales_order_id;
                    }

                    foreach ($attributes['sales_credit_lines'] as $index => $creditLine) {
                        if (isset($creditLine['sales_order_line_id'])) {
                            /** @var SalesOrderLine $salesOrderLine */
                            $salesOrderLine = $salesOrderLines->firstWhere('id', $creditLine['sales_order_line_id']);

                            if ($salesOrderId) {
                                if ($salesOrderId && $salesOrderLine->sales_order_id != $salesOrderId) {
                                    $validator->addFailure("sales_credit_lines.{$index}.sales_order_line_id", 'IsNotBelongsToSalesOrder');
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_not_belongs_to_sales_order' => __('messages.sales_order.is_not_belongs'),
            'is_returned' => __('messages.sales_order.fully_returned_line'),
            'is_not_fulfilled' => __('messages.sales_order.is_not_fulfilled'),
            'is_greater_than_unreturned_quantity' => __('messages.sales_order.greater_than_unreturned'),
            'is_not_closed' => __('messages.sales_order.is_not_closed'),
        ];
    }
}
