<?php

namespace App\Http\Requests;

use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class StorePurchaseOrderShipment extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        if ($this->isMethod('POST')) {
            return [
                'purchase_order_id' => 'required|exists:purchase_orders,id',
                'shipment_date' => 'required|date',
                'shipping_method_id' => 'nullable|exists:shipping_methods,id',
                'tracking' => 'nullable|max:255',

                'shipment_lines' => 'nullable|array',
                'shipment_lines.*.purchase_order_line_id' => 'required_without:shipment_lines.*.purchase_order_line_reference|exists:purchase_order_lines,id',
                'shipment_lines.*.quantity' => 'required|numeric',
                'shipment_lines.*.purchase_order_line_reference' => 'required_without:shipment_lines.*.purchase_order_line_id|exists:purchase_order_lines,line_reference',
            ];
        }

        // Update PO Shipment
        return [
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'tracking' => 'nullable|max:255',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // Purchase order shouldn't be draft
                $purchaseOrder = PurchaseOrder::with([
                    'purchaseOrderLines',
                    'purchaseOrderShipments',
                    'purchaseOrderShipmentLines',
                ])
                    ->findOrFail($attributes['purchase_order_id']);
                if ($purchaseOrder->isDraft()) {
                    $validator->addFailure('purchase_order', 'IsNotOpen');

                    return;
                }

                // check purchase order lines belong to the purchase order and unfulfilled
                if (! empty($attributes['shipment_lines'])) {
                    // Aggregate shipment lines quantities based on id's.
                    // This ensures that the user does not create more shipment
                    // quantities than on the quantity unfulfilled
                    $attributes['shipment_lines'] = PurchaseOrderShipment::aggregateShipmentLines($attributes['shipment_lines']);

                    $purchaseOrderLines = PurchaseOrderLine::with(['purchaseOrderShipmentLines'])
                        ->whereIn('id', array_column($attributes['shipment_lines'], 'purchase_order_line_id'))
                        ->get();

                    foreach ($attributes['shipment_lines'] as $index => $shipmentLine) {
                        $hasLineReference = isset($shipmentLine['purchase_order_line_reference']);
                        $key = ! $hasLineReference ? 'purchase_order_line_id' : 'purchase_order_line_reference';

                        /** @var PurchaseOrderLine $purchaseOrderLine */
                        $purchaseOrderLine = $purchaseOrderLines->firstWhere('id', $shipmentLine['purchase_order_line_id']);

                        if ($purchaseOrderLine->purchase_order_id != $attributes['purchase_order_id']) {
                            $validator->addFailure("shipment_lines.{$index}.{$key}", 'IsNotBelongsToPurchaseOrder');
                        }

                        if ($purchaseOrderLine->fully_shipped) {
                            $validator->addFailure("shipment_lines.{$index}.{$key}", 'IsFulfilled');
                        } elseif ($shipmentLine['quantity'] > $purchaseOrderLine->unfulfilled_quantity) {
                            $validator->addFailure("shipment_lines.{$index}.quantity", 'IsGreaterThanUnfulfilledQuantity');
                        }
                    }
                } else {
                    // No lines provided, user is attempting to ship everything
                    // We check to make sure the purchase order isn't fully shipped.
                    if ($purchaseOrder->fully_shipped) {
                        $validator->addFailure('purchase_order', 'IsFullyShipped');
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_not_belongs_to_purchase_order' => __('messages.purchase_order.is_not_belongs'),
            'is_fulfilled' => __('messages.purchase_order.fully_fulfilled_line'),
            'is_not_open' => __('messages.purchase_order.is_not_open'),
            'is_fully_shipped' => __('messages.purchase_order.fully_shipped'),
            'is_greater_than_unfulfilled_quantity' => __('messages.purchase_order.greater_than_unfulfilled'),
        ];
    }
}
