<?php

namespace App\Http\Requests;

use App\Models\Product;
use App\Models\Warehouse;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Validator;

class StoreInventoryAssembly extends FormRequest
{
    use InstanceFromCustom;

    const ACTION_TYPE_ASSEMBLE = 'assemble';

    const ACTION_TYPE_DISASSEMBLE = 'disassemble';

    const ACTION_TYPES = [
        self::ACTION_TYPE_ASSEMBLE,
        self::ACTION_TYPE_DISASSEMBLE,
    ];

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'action_date' => 'required|date',
            'product_id' => 'required|exists:products,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'warehouse_location_id' => 'nullable|exists:warehouse_locations,id',
            'quantity' => 'required|numeric',
            'action' => 'required|in:'.implode(',', self::ACTION_TYPES),
        ];

        if ($this->isMethod('put')) {
            $rules['adjustment_date'] = 'sometimes|'.$rules['adjustment_date'];
            $rules['warehouse_id'] = 'sometimes|'.$rules['warehouse_id'];
            $rules['quantity'] = 'sometimes|'.$rules['quantity'];
            $rules['action'] = 'sometimes|'.$rules['action'];
            unset($rules['product_id']); // Product cannot be changed for the assembly
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (\App\Validator $validator) {
                $attributes = $validator->attributes();

                if ($this->isMethod('PUT')) {
                    // Updating the assembly
                    $this->validateExtraForUpdate($validator, $attributes);
                } else {
                    // Creating assemblies
                    $this->validateExtraForPost($validator, $attributes);
                }
            });
        }
    }

    private function productHasAverageCost($productId): bool
    {
        $product = Product::with([])->findOrFail($productId);

        return (bool) $product->average_cost;
    }

    private function warehouseHasDefaultLocation($warehouseId): bool
    {
        $warehouse = Warehouse::with(['defaultLocation'])->findOrFail($warehouseId);

        return ! empty($warehouse->defaultLocation);
    }

    private function validateExtraForUpdate(Validator $validator, array $data)
    {
        // If changing the warehouse, it must have a default location
        if (isset($data['warehouse_id']) && ! $this->warehouseHasDefaultLocation($data['warehouse_id'])) {
            $validator->addFailure('warehouse_id', 'NeedsDefaultLocation');
        }
    }

    private function validateExtraForPost(Validator $validator, array $assembly)
    {
        // Check for non-zero quantity for action
        if ($assembly['quantity'] == 0) {
            $validator->addFailure('quantity', 'InvalidQuantity');
        }

        if (empty($assembly['warehouse_location_id'])) {
            if (! $this->warehouseHasDefaultLocation($assembly['warehouse_id'])) {
                $validator->addFailure('warehouse_id', 'NeedsDefaultLocation');
            }
        }
    }

    public function messages(): array
    {
        return [
            'needs_default_location' => __('messages.warehouse.warehouse_needs_location'),
            'invalid_quantity' => __('messages.inventory.adjustment_quantity_not_zero'),
        ];
    }

    public function mergeToData($data)
    {
        $this->merge($data);

        $this->getValidatorInstance()->addToData($data);
    }
}
