<?php

namespace App\Http\Requests;

use App\Models\Attribute;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Validator;

class StoreAttributeGroup extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'name' => [
                'required',
                'max:255',
                Rule::unique('attribute_groups', 'name')
                    ->ignore($this->route('attribute_group')),
            ],
            'sort_order' => 'nullable|integer',
            'attributes' => 'nullable|array',
            'attributes.*.id' => 'required|exists:attributes,id',
            'attributes.*.sort_order' => 'nullable|integer',
        ];

        if ($this->isMethod('PUT')) {
            $rules['name'] = array_merge(['sometimes'], $rules['name']);
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (\App\Validator $validator) {
                $attributes = $validator->attributes();

                if (! empty($attributes['attributes'])) {
                    $selectedAttributes = Attribute::with([])->whereIn('id', array_column($attributes['attributes'], 'id'))->get();

                    foreach ($attributes['attributes'] as $index => $attribute) {
                        /** @var Attribute $selectedAttribute */
                        $selectedAttribute = $selectedAttributes->firstWhere('id', $attribute['id']);
                        if ($selectedAttribute->attribute_group_id && $selectedAttribute->attribute_group_id != $this->route('attribute_group')) {
                            $validator->addFailure("attributes.$index.id", 'AssignedToAnotherAttributeGroup', []);
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'assigned_to_another_attribute_group' => __('messages.attribute_group.assign_attribute_failed'),
        ];
    }
}
