<?php

namespace App\Http\Requests;

use App\Models\Address;
use App\Models\Customer;
use App\Models\Supplier;
use App\Validator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Http\FormRequest;

/**
 * Class StoreAddressRequest.
 */
class StoreAddressRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'name' => 'nullable|max:255',
            'company' => 'nullable|max:255',
            'email' => 'nullable|email',
            'phone' => 'nullable',
            'fax' => 'nullable|max:255',
            'address1' => 'nullable|max:255',
            'address2' => 'nullable|max:255',
            'address3' => 'nullable|max:255',
            'city' => 'nullable|max:255',
            'province' => 'nullable|max:255',
            'province_code' => 'nullable|max:255',
            'zip' => 'nullable|max:255',
            'country' => 'nullable|max:255',
            'country_code' => 'nullable|max:2',
            'label' => 'required|max:255',
            'is_default_shipping' => 'nullable|boolean',
            'is_default_billing' => 'nullable|boolean',
        ];

        if ($this->isMethod('PUT')) {
            $rules['label'] = 'sometimes|'.$rules['label'];
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                if (! empty($attributes['label']) || ! empty($attributes['email'])) {
                    // We get the owner of the address (addressable) from the route
                    $addressable = $this->findAddressableFromRoute();

                    if ($addressable != null) {
                        // check label is unique by customer
                        if (! empty($attributes['label'])) {
                            $addresses = collect($addressable->all_addresses)->firstWhere('label', $attributes['label']);

                            if ($addresses) {
                                if ($this->isMethod('POST') ||
                     ($this->isMethod('PUT') && $addresses['id'] != $this->route()->originalParameter('address'))) {
                                    $validator->addFailure('label', $this->getFailureRule());
                                }
                            }
                        }

                        // check email in between customers
                        if (! empty($attributes['email']) && $this->route('customer')) {
                            $emailExists = Customer::with([])->where('email', $attributes['email'])->where('id', '!=', $addressable->id)->exists();
                            if ($emailExists) {
                                $validator->addFailure('email', 'EmailExistsInCustomerAddresses');
                            } else {
                                $emailExists = Address::with([])
                                    ->where('email', $attributes['email'])
                                    ->where(function (Builder $builder) use ($addressable) {
                                        $builder
                                            ->whereHas('customerByShippingAddress', function (Builder $builder) use ($addressable) {
                                                $builder->where('customers.id', '!=', $addressable->id);
                                            })
                                            ->orWhereHas('customerByBillingAddress', function (Builder $builder) use ($addressable) {
                                                $builder->where('customers.id', '!=', $addressable->id);
                                            })
                                            ->orWhereHas('customerByOtherAddresses', function (Builder $builder) use ($addressable) {
                                                $builder->where('customers.id', '!=', $addressable->id);
                                            });
                                    })->exists();
                                if ($emailExists) {
                                    $validator->addFailure('email', 'EmailExistsInCustomerAddresses');
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'must_be_unique_by_customer' => __('messages.customer.label_unique'),
            'email_exists_in_customer_addresses' => __('messages.customer.email_exists'),
            'must_be_unique_by_supplier' => __('messages.supplier.label_unique'),
        ];
    }

    /**
     * Finds addressable based on the route parameter.
     *
     * @return \Illuminate\Database\Eloquent\Collection|\Illuminate\Database\Eloquent\Model|mixed|null|static|static[]
     */
    private function findAddressableFromRoute()
    {
        $addressable = null;

        if ($this->route('customer')) {
            // We attempt to find customer or fail
            $addressable = Customer::with([
                'shippingAddress',
                'billingAddress',
                'addresses',
            ])->findOrFail($this->route('customer'));
        } elseif ($this->route('supplier')) {
            // We attempt to find supplier or fail
            $addressable = Supplier::with(['addresses'])->findOrFail($this->route('supplier'));
        } elseif ($address = $this->route('address')) {
            // We only have an address, we get the linked addressable for unique label checking
            $addressable = $this->getAddressableForAddress($address);
        }

        return $addressable;
    }

    /**
     * Gets addressable for given address.
     *
     *
     * @return mixed|null
     */
    private function getAddressableForAddress(Address $address)
    {
        // Check if it's supplier
        if (($suppliers = $address->suppliers)->count() > 0) {
            return $suppliers->first();
        }
        // Check if it's customer
        if ($customer = $this->getCustomerForAddress($address)) {
            return $customer;
        }

        // Check other addressables here such as warehouses, etc
        return null;
    }

    /**
     * Gets customer for given address.
     *
     *
     * @return mixed|null
     */
    private function getCustomerForAddress(Address $address)
    {
        $customer = null;
        $address->load('customerByShippingAddress', 'customerByBillingAddress', 'customerByOtherAddresses');

        if ($address->customerByShippingAddress) {
            $customer = $address->customerByShippingAddress;
        } elseif ($address->customerByBillingAddress) {
            $customer = $address->customerByBillingAddress;
        } elseif ($address->customerByOtherAddresses->isNotEmpty()) {
            $customer = $address->customerByOtherAddresses->first();
        }

        return $customer;
    }

    protected function getFailureRule(): string
    {
        if ($address = $this->route('address')) {
            $addressable = $this->getAddressableForAddress($address);
        }

        if ($this->route('customer') || (isset($addressable) && $addressable instanceof Customer)) {
            return 'MustBeUniqueByCustomer';
        }
        if ($this->route('supplier') || (isset($addressable) && $addressable instanceof Supplier)) {
            return 'MustBeUniqueBySupplier';
        }

        return '';
    }
}
