<?php

namespace App\Http\Requests;

use App\Models\SalesOrder;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class ResendSalesOrderRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'sales_order_lines' => 'nullable|array',
            'sales_credit_id' => 'exists:sales_credits,id',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();
                /** @var SalesOrder $salesOrder */
                $salesOrder = $this->route('sales_order');
                $salesOrder->load('salesOrderLines');

                if (isset($attributes['sales_order_lines'])) {
                    foreach ($attributes['sales_order_lines'] as $index => $line) {
                        $salesOrderLine = $salesOrder->salesOrderLines->firstWhere('id', $line['id']);
                        if (! $salesOrderLine) {
                            $validator->addFailure("sales_order_lines.$index.id", 'IsNotBelongsToOrder');
                        } elseif ($this->routeIs('sales-orders.resend') && $line['quantity'] > $salesOrderLine->quantity) {
                            $validator->addFailure("sales_order_lines.$index.quantity", 'IsGreaterThanOriginalQuantity');
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_not_belongs_to_order' => __('messages.sales_order.line_not_exists_in_order'),
            'is_greater_than_original_quantity' => __('messages.sales_order.quantity_greater_original'),
        ];
    }
}
