<?php

namespace App\Http\Requests;

use App\Models\ProductCategory;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class ReassignCategoryToProducts extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'old_category_id' => 'required',
            'new_category_id' => 'required|different:old_category_id',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                $categories = ProductCategory::with([])->whereIn('id', [
                    $attributes['old_category_id'],
                    $attributes['new_category_id'],
                ])->get();

                // check an old category id exists
                if (! $categories->firstWhere('id', $attributes['old_category_id'])) {
                    $validator->addFailure('old_category_id', 'Exists', []);
                }

                // check a new category id exists
                if (! $categories->firstWhere('id', $attributes['new_category_id'])) {
                    $validator->addFailure('new_category_id', 'Exists', []);
                }

                // check there are a non-leaf categories
                foreach ($categories as $category) {
                    $validationAttribute = $category->id == $attributes['old_category_id'] ? 'old_category_id' : 'new_category_id';

                    if (! $category->is_leaf) {
                        $validator->addFailure($validationAttribute, 'IsNonLeaf', []);
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_non_leaf' => __('messages.category.non-leaf_failed'),
        ];
    }
}
