<?php

namespace App\Http\Requests;

use App\Exceptions\ActionUnavailableTemporaryException;
use Illuminate\Contracts\Cache\Lock;
use Illuminate\Support\Facades\Cache;
use Throwable;

trait LockRequestTrait
{
    protected Lock $requestLock;

    /**
     * Return the id of the lock
     */
    abstract protected function lockId(): string;

    /**
     * Lock the request
     *
     * @throws ActionUnavailableTemporaryException
     */
    protected function lock(): void
    {
        $this->requestLock = Cache::driver('redis')->lock(static::class.':'.$this->lockId(), 60);
        if (! app()->runningUnitTests() && ! $this->requestLock->get()) {
            // already locked by another request
            throw new ActionUnavailableTemporaryException();
        }
    }

    /**
     * @throws ActionUnavailableTemporaryException|Throwable
     */
    public function validateResolved(): void
    {
        try {
            // lock the request
            $this->lock();

            parent::validateResolved();
        } finally {
            isset($this->requestLock) && $this->requestLock->release();
        }
    }

    public function __destruct()
    {
        // when destructing the request release the lock
        isset($this->requestLock) && $this->requestLock->release();
    }
}
