<?php

namespace App\Http\Requests;

use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Response;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

/**
 * Class FulfillFBASalesOrderRequest.
 */
class FulfillFBASalesOrderRequest extends FormRequest
{
    use InstanceFromCustom;

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'warehouse_id' => ['nullable', Rule::exists('warehouses', 'id')->where('type', Warehouse::TYPE_AMAZON_FBA)],
            'shipping_speed' => 'required|in:Standard,Expedited,Priority',
            'comments' => 'required',
            'fulfillment_lines' => 'array|min:1',
            'fulfillment_lines.*.sales_order_line_id' => 'required|exists:sales_order_lines,id',
            'fulfillment_lines.*.quantity' => 'required|numeric|min:1',
            'fulfillment_lines.*.product_listing_id' => 'required|exists:product_listings,id',
        ];

        if ($this->isMethod('PUT')) {
            unset($rules['warehouse_id']);
            // send fulfillment line id or sales order id
            $rules['fulfillment_lines.*.id'] = 'required_without:fulfillment_lines.*.sales_order_line_id|exists:sales_order_fulfillment_lines,id';
            $rules['fulfillment_lines.*.sales_order_line_id'] = 'required_without:fulfillment_lines.*.id|exists:sales_order_lines,id';
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // check sales order is exists and approved
                /** @var SalesOrder $salesOrder */
                $salesOrder = $this->isMethod('POST') ? $this->route('sales_order') : $this->route('sales_order_fulfillment')->salesOrder;
                // sales order status must be open
                $this->checkSalesOrderStatus($salesOrder, $validator);

                // check sales order lines belong to sales order and unfulfilled
                $salesOrder->load('salesOrderLines', 'salesOrderLines.salesOrderFulfillmentLines');

                if ($this->isMethod('POST')) {
                    $productWarehouses = $salesOrder->salesOrderLines->where('is_product', true)->pluck('warehouse_id')->unique();

                    if ($productWarehouses->count() === 1) {
                        $attributes['warehouse_id'] = $productWarehouses->first();
                        $this->getValidatorInstance()->addToData(['warehouse_id' => $attributes['warehouse_id']]);
                    } else {
                        if (empty($attributes['warehouse_id'])) {
                            $validator->addFailure('warehouse_id', 'Required');
                        } elseif (! $productWarehouses->firstWhere(null, $attributes['warehouse_id'])) {
                            $validator->addFailure('warehouse_id', 'IsNotBelongsToSalesOrderWarehouses');
                        }
                    }

                    if (! empty($attributes['warehouse_id']) && empty($attributes['fulfillment_lines'])) {
                        $unfulfilledLines = [];
                        foreach ($salesOrder->salesOrderLines as $line) {
                            if ($line->warehouse_id == $attributes['warehouse_id'] && $line->is_product && $line->unfulfilled_quantity) {
                                $unfulfilledLines[] = ['sales_order_line_id' => $line->id, 'quantity' => $line->unfulfilled_quantity];
                            }
                        }
                        $attributes['fulfillment_lines'] = $unfulfilledLines;
                        $this->getValidatorInstance()->addToData(['fulfillment_lines' => $attributes['fulfillment_lines']]);
                    }
                }

                foreach ($attributes['fulfillment_lines'] ?? [] as $index => $fulfillmentLine) {
                    /** @var SalesOrderLine $salesOrderLine */
                    $salesOrderLine = $salesOrder->salesOrderLines->firstWhere('id', $fulfillmentLine['sales_order_line_id']);

                    if (empty($salesOrderLine) || $salesOrderLine->sales_order_id != $salesOrder->id) {
                        $validator->addFailure("fulfillment_lines.{$index}.sales_order_line_id", 'IsNotBelongsToSalesOrder');

                        continue;
                    }

                    if ($this->isMethod('POST')) {
                        if ($salesOrderLine->fulfilled) {
                            $validator->addFailure("fulfillment_lines.{$index}.sales_order_line_id", 'IsFulfilled');
                        } elseif ($fulfillmentLine['quantity'] > $salesOrderLine->unfulfilled_quantity) {
                            $validator->addFailure("fulfillment_lines.{$index}.quantity", 'IsGreaterThanUnfulfilledQuantity');
                        }
                    }

                    if (isset($attributes['warehouse_id']) && $salesOrderLine->warehouse_id != $attributes['warehouse_id']) {
                        $validator->addFailure("fulfillment_lines.{$index}.sales_order_line_id", 'IsNotBelongsToWarehouse');
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_not_approved' => __('messages.sales_order.is_not_approved'),
            'is_closed' => __('messages.sales_order.is_closed'),
            'is_not_belongs_to_sales_order' => __('messages.sales_order.is_not_belongs'),
            'is_not_belongs_to_sales_order_warehouses' => __('messages.sales_order.is_not_belongs_warehouses'),
            'is_fulfilled' => __('messages.sales_order.fully_fulfilled'),
            'is_greater_than_unfulfilled_quantity' => __('messages.sales_order.greater_than_unfulfilled_quantity'),
            'is_not_belongs_to_warehouse' => __('messages.sales_order.is_not_belongs_to_warehouse'),
        ];
    }

    /**
     * Determine if the sales order status is open.
     */
    private function checkSalesOrderStatus(SalesOrder $salesOrder, Validator $validator)
    {
        if ($salesOrder->order_status == SalesOrder::STATUS_DRAFT) {
            $validator->addFailure('sales_order_id', Response::CODE_NOT_APPROVED);
        }

        if ($salesOrder->order_status == SalesOrder::STATUS_CLOSED) {
            $validator->addFailure('sales_order_id', Response::CODE_CLOSED);
        }
    }
}
