<?php

namespace App\Http\Controllers;

use App\Data\WarehouseTransferReceiptData;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Http\Requests\WarehouseTransferShipmentReceiptRequest;
use App\Http\Resources\WarehouseTransferReceiptResource;
use App\Http\Resources\WarehouseTransferResource;
use App\Managers\WarehouseTransferManager;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Repositories\WarehouseRepository;
use App\Response;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Throwable;

class WarehouseTransferShipmentController extends Controller
{
    /**
     * @var WarehouseTransferManager
     */
    private $transferManager;

    /**
     * @var WarehouseRepository
     */
    private $warehouses;

    /**
     * WarehouseTransferShipmentController constructor.
     */
    public function __construct(WarehouseTransferManager $transferManager, WarehouseRepository $warehouses)
    {
        parent::__construct();
        $this->transferManager = $transferManager;
        $this->warehouses = $warehouses;
    }

    public function receipts($transferId)
    {
        $transfer = WarehouseTransfer::with(['shipmentReceipts'])->findOrFail(e($transferId));
        $receipts = $transfer->shipmentReceipts()->with(['receiptLines', 'receiptLines.shipmentLine'])->get();

        return $this->response->addData(WarehouseTransferReceiptResource::collection($receipts))->success();
    }

    /**
     * @throws Throwable
     */
    public function receiveShipment(WarehouseTransferReceiptData $data, WarehouseTransfer $transfer): Response
    {
        try {
            // We create the shipment using the transfer manager
            $result = $this->transferManager->receiveShipment($transfer, $data);
            $transfer = $result instanceof WarehouseTransfer ? $result : $result->shipment->warehouseTransfer;
            $transfer
                ->load(
                    DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class)
                )
                ->loadMissing('shipmentReceipts');

            return $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.create', ['resource' => 'warehouse transfer shipment receipt']))
                ->addData(WarehouseTransferResource::make($transfer));
        } catch (NotOpenWarehouseTransferException $e) {
            // The warehouse transfer is not open yet for shipment to be created.
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_status_not_open', [
                    'transfer_id' => $e->transfer->id,
                    'extra' => '',
                ]), 'WarehouseTransfer'.Response::CODE_IS_NOT_OPEN, 'transfer_id', ['transfer_id' => $e->transfer->id]);
        } catch (Exception $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setMessage($e->getMessage());
        }
    }

    /**
     * @param  WarehouseTransferShipmentReceiptRequest  $request
     * @param $transferId
     * @param $receiptId
     * @return Response
     */
    public function updateReceipt(WarehouseTransferShipmentReceiptRequest $request, $transferId, $receiptId)
    {
        // Get the warehouse transfer and receipt
        /** @var WarehouseTransfer $warehouseTransfer */
        $warehouseTransfer = WarehouseTransfer::with([])->findOrFail(e($transferId));
        $relations = [
            'receiptLines',
            'receiptLines.shipmentLine',
            'receiptLines.shipmentLine.warehouseTransferLine',
            'receiptLines.shipmentLine.warehouseTransferLine.product',
        ];
        $receipt = WarehouseTransferShipmentReceipt::with($relations)->findOrFail(e($receiptId));

        try {
            // Update the shipment using the transfer manager
            $warehouseTransfer = $this->transferManager->updateShipmentReceipt($warehouseTransfer, $receipt, $request->validated());
            $warehouseTransfer->load(DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class));

            return $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.update', ['resource' => 'warehouse transfer shipment receipt']));
        } catch (NotOpenWarehouseTransferException $e) {
            // The warehouse transfer is not open yet for shipment to be created.
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_status_not_open', [
                    'transfer_id' => $e->transfer->id,
                    'extra' => '',
                ]), 'WarehouseTransfer'.Response::CODE_IS_NOT_OPEN, 'transfer_id', ['transfer_id' => $e->transfer->id]);
        } catch (Throwable $e) {
            return $this->response->error()
                ->setMessage($e->getMessage());
        }
    }

    public function deleteReceipt($transferId, $receiptId)
    {
        // Get the warehouse transfer and receipt
        $warehouseTransfer = WarehouseTransfer::with([
            'shipmentLines',
            'shipmentLines.receiptLines',
        ])->findOrFail(e($transferId));
        $receipt = WarehouseTransferShipmentReceipt::with(['shipment'])->whereHas('shipment', function (Builder $builder) use ($transferId) {
            return $builder->where('warehouse_transfer_id', $transferId);
        })->findOrFail(e($receiptId));

        try {
            // Update the shipment using the transfer manager
            $warehouseTransfer = $this->transferManager->deleteShipmentReceipt($warehouseTransfer, $receipt);
            $warehouseTransfer->load(DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class));

            return $this->response->success(Response::HTTP_CREATED)
                ->addData(WarehouseTransferResource::make($warehouseTransfer))
                ->setMessage(__('messages.success.delete', [
                    'resource' => 'warehouse transfer shipment receipt',
                    'id' => $receipt->id,
                ]));
        } catch (NotOpenWarehouseTransferException $e) {
            // The warehouse transfer is not open yet for shipment to be created.
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_status_not_open', [
                    'transfer_id' => $e->transfer->id,
                    'extra' => '',
                ]), 'WarehouseTransfer'.Response::CODE_IS_NOT_OPEN, 'transfer_id', ['transfer_id' => $e->transfer->id]);
        } catch (Exception $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setMessage($e->getMessage());
        }
    }
}
