<?php

namespace App\Http\Controllers;

use App\Models\Warehouse;
use App\Models\WarehouseLocation;
use App\Response;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class WarehouseLocationController extends Controller
{
    public function index(Warehouse $warehouse)
    {
        return $this->response->addData($warehouse->warehouseLocations);
    }

    /**
     * Create new warehouse location.
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'warehouse_id' => 'required|exists:warehouses,id',
            'aisle' => 'required|max:255',
            'bay' => 'nullable|max:255',
            'shelf' => 'nullable|max:255',
            'bin' => 'nullable|max:255',
            'is_default' => 'nullable|boolean',
        ]);

        $inputs = $request->all();

        $warehouseLocation = new WarehouseLocation($inputs);
        $warehouseLocation->warehouse_id = $inputs['warehouse_id'];
        $warehouseLocation->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => 'warehouse location']))
            ->addData($warehouseLocation);
    }

    /**
     * Update warehouse location.
     */
    public function update(Request $request, $warehouse_location_id): JsonResponse
    {
        $request->validate([
            'aisle' => 'nullable|max:255',
            'bay' => 'nullable|max:255',
            'shelf' => 'nullable|max:255',
            'bin' => 'nullable|max:255',
            'is_default' => 'nullable|boolean',
        ]);

        $inputs = array_filter($request->all());

        $warehouseLocation = WarehouseLocation::with([])->findOrFail($warehouse_location_id);
        $warehouseLocation->fill($inputs);
        $warehouseLocation->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => 'warehouse location',
                'id' => $warehouse_location_id,
            ]))
            ->addData($warehouseLocation);
    }
}
