<?php

namespace App\Http\Controllers;

use App\Models\ProductInventory;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\Warehouse;
use App\Response;
use Illuminate\Http\Request;

/**
 * Class WarehouseDashboardController.
 */
class WarehouseDashboardController extends Controller
{
    /**
     * Gets the inventory summary of the warehouse.
     */
    public function inventorySummary($warehouseId): Response
    {
        $query = ProductInventory::with([])
            ->where('warehouse_id', e($warehouseId));
        $unitsAvailable = $query->sum('inventory_available');
        $valuation = $query->sum('inventory_stock_value');

        return $this->response->addData(['units_available' => $unitsAvailable, 'valuation' => $valuation]);
    }

    /**
     * Gets the inventory activity for the warehouse.
     */
    public function inventoryActivity($warehouseId): Response
    {
        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::with([])->findOrFail(e($warehouseId));

        $activity = $warehouse->inventoryMovements()->select('product_id')->groupBy('product_id')->count();
        $last7Days = $warehouse->inventoryMovements()
            ->whereDate('inventory_movement_date', '>=', now()->subDays(7))
            ->select('product_id')
            ->groupBy('product_id')
            ->count();

        $last30Days = $warehouse->inventoryMovements()
                            ->whereDate('inventory_movement_date', '>=', now()->subDays(30))
                            ->select('product_id')
                            ->groupBy('product_id')
                            ->count();

        return $this->response->addData([
            'activity' => $activity,
            'past_week' => $last7Days,
            'past_month' => $last30Days,
        ]);
    }

    public function inventoryShipments(Request $request, $warehouseId)
    {
        // We get the purchase order shipment receipts for the warehouse.
        $from = $request->query('from');
        $to = $request->query('to');
        $days = $request->get('days', 1);

        $query = SalesOrderFulfillmentLine::query()
            ->leftJoin('sales_order_fulfillments', 'sales_order_fulfillments.id', 'sales_order_fulfillment_lines.sales_order_fulfillment_id')
            ->where('sales_order_fulfillments.warehouse_id', $warehouseId);

        if ($from) {
            $query->whereDate('sales_order_fulfillments.created_at', '>=', $from);
        }

        if ($to) {
            $query->whereDate('sales_order_fulfillments.created_at', '<=', $to);
        }

        if ($days) {
            $data = $query->groupByRaw('DATE(`sales_order_fulfillments`.`created_at`)')
                ->selectRaw('DATE(`sales_order_fulfillments`.`created_at`) as day, SUM(`quantity`) as quantity')
                ->get();

            return $this->response->addData($data);
        } else {
            return $this->response->addData(['quantity' => $query->sum('quantity')]);
        }
    }
}
