<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\AddressLockedException;
use App\Exceptions\DropshipWithOpenOrdersException;
use App\Exporters\InitialInventoryExporter;
use App\Helpers;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreWarehouse;
use App\Http\Resources\PurchaseOrderResource;
use App\Http\Resources\SupplierWarehouseResource;
use App\Http\Resources\WarehouseResource;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Repositories\SupplierInventoryRepository;
use App\Repositories\WarehouseRepository;
use App\Response;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class WarehouseController extends Controller
{
    use bulkOperation
  {
      archiveInstance as originalArchiveInstance;
  }
    use DataTable, ImportsData;

    protected $model_path = Warehouse::class;

    protected $resource = 'warehouse';

    /**
     * @var WarehouseRepository
     */
    protected $warehouses;

    /**
     * @var SupplierInventoryRepository
     */
    protected $inventories;

    /**
     * WarehouseController constructor.
     */
    public function __construct(WarehouseRepository $warehouses, SupplierInventoryRepository $inventories)
    {
        $this->warehouses = $warehouses;
        $this->inventories = $inventories;
        parent::__construct();
    }

    public function index(Request $request)
    {
        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications($this->getModel()));
        }

        // prevent send included and excluded together
        if ($request->has('included') and $request->has('excluded')) {
            return $this->response->error(Response::HTTP_BAD_REQUEST, [__('messages.failed.not_both_include_exclude')])
                ->setMessage(__('messages.failed.not_both_include_exclude'));
        }

        // set default included
        $this->setDefaultIncludedColumns($request);

        /**
         * model with relations.
         */
        $builder = Warehouse::with(['address']);

        $builder->filter();
        $builder->addRelations();
        $builder->addCustomColumns();
        $builder->sort();
        $builder->archived($request->get('archived', 0));

        // without supplier warehouses
        if (boolval($request->get('exclude_supplier_warehouses', 1))) {
            //This is not a sufficient condition
            //$builder->whereNull('supplier_id');
            $builder->where('type', '!=', Warehouse::TYPE_SUPPLIER);
        } else
        {
            if ($product_id = $request->get('suppliers_for_product_id'))
            {
                $product = Product::find($product_id);
                $builder->where(function (Builder $query) use ($product) {
                    $query->where(function (Builder $query) use ($product) {
                        $query->where('type', '=', Warehouse::TYPE_SUPPLIER)
                            ->whereIn('supplier_id', $product->supplierProducts->pluck('supplier_id')->toArray());
                    });
                    $query->orWhere('type', '!=', Warehouse::TYPE_SUPPLIER);
                });

            }
        }

        // paginate with limit per page (default 10)
        $warehouses = DataTableConfiguration::paginate($builder);

        return WarehouseResource::collectionWithTableSpecifications($warehouses, Warehouse::class);
    }

    public function show(Warehouse $warehouse)
    {
        $warehouse->load('address', 'warehouseLocations');

        // Load the supplier if it's supplier warehouse
        if ($warehouse->isSupplierWarehouse()) {
            $warehouse->load('supplier');

            return $this->response->addData(SupplierWarehouseResource::make($warehouse));
        }

        return $this->response->addData(WarehouseResource::make($warehouse));
    }

    /**
     * Create a new warehouse.
     * @throws DropshipWithOpenOrdersException
     */
    public function store(StoreWarehouse $request): Response
    {
        $warehouse = $this->warehouses->createWarehouse($request->validated());
        $warehouse->load('address');

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(WarehouseResource::make($warehouse));
    }

    /**
     * Update the warehouse.
     */
    public function update(StoreWarehouse $request, $warehouse_id): Response
    {
        return DB::transaction(function () use ($request, $warehouse_id) {
            $inputs = $request->validated();

            $warehouse = Warehouse::with(['defaultLocation'])->findOrFail($warehouse_id);

            if ($warehouse->isSupplierWarehouse() && isset($inputs['is_default']) && $inputs['is_default']) {
                // This must be the default warehouse of the supplier
                $warehouse->supplier->setDefaultWarehouse($warehouse_id);

                // If the supplier warehouse doesn't have inventory initiated, we do that now.
                if ($warehouse->supplierInventory()->count() === 0) {
                    $this->inventories->initializeInventoryForSupplierWarehouse($warehouse);
                }
            }

            // Update the warehouse
            $warehouse->fill($inputs);
            try {
                $warehouse->save();
                if (! $warehouse->isSupplierWarehouse()) {
                    $warehousePriority = json_decode(Helpers::setting(Setting::KEY_WAREHOUSE_PRIORITY), true) ?: [];
                    $indexes = array_keys($warehousePriority, $warehouse->id);
                    if ($warehouse->auto_routing_enabled) {
                        $indexes = array_keys($warehousePriority, $warehouse->id);
                        if (empty($indexes)) {
                            $warehousePriority[] = $warehouse->id;
                        }
                    } else {
                        $indexes = array_keys($warehousePriority, $warehouse->id);
                        foreach ($indexes as $index) {
                            unset($warehousePriority[$index]);
                        }
                    }

                    $setting = Setting::where('key', Setting::KEY_WAREHOUSE_PRIORITY)->firstOrFail();
                    $setting->value = json_encode(array_values($warehousePriority));
                    $setting->save();
                }
            } catch (DropshipWithOpenOrdersException $e) {
                return $this->response->error(Response::HTTP_BAD_REQUEST)
                    ->addError(__('messages.warehouse.dropship_has_orders', ['id' => $warehouse->name]), 'UpdateWarehouse'.Response::CODE_UNACCEPTABLE, 'id', ['id' => $warehouse->id]);
            }

            if (empty($warehouse->defaultLocation)) {
                if (empty($inputs['default_location'])) {
                    $inputs['default_location'] = ['aisle' => 'Default'];
                }
                $warehouse->defaultLocation()->create(array_merge($inputs['default_location'], ['is_default' => true]));
            }

            // Set the warehouse address
            try {
                if(isset($inputs['address_name'])) {
                    $this->warehouses->setWarehouseAddress($warehouse, $inputs);
                }
            } catch (AddressLockedException $e)
            {
                return $this->response->error(Response::HTTP_BAD_REQUEST)
                    ->addError('Warehouse address is locked due to usage on sales orders and cannot be changed', 500, 'id', ['id' => $warehouse->id]);
            }

            $warehouse->load('address', 'warehouseLocations');

            // Load supplier info if is supplier warehouse
            $isSupplierWarehouse = $warehouse->isSupplierWarehouse();
            if ($isSupplierWarehouse) {
                $warehouse->load('supplier');
            }

            return $this->response
                ->setMessage(__('messages.success.update', [
                    'resource' => $this->resource,
                    'id' => $warehouse->name,
                ]))
                ->addData($isSupplierWarehouse ? SupplierWarehouseResource::make($warehouse) : WarehouseResource::make($warehouse));
        });
    }

    public function incomingPurchaseOrders($warehouseId): Response
    {
        $warehouse = Warehouse::with(['purchaseOrders'])->findOrFail(e($warehouseId));
        $purchaseOrders = $warehouse->purchaseOrders()->with([
            'purchaseOrderLines',
            'supplier',
        ])->where('order_status', PurchaseOrder::STATUS_OPEN)->get();

        return $this->response->addData(PurchaseOrderResource::collection($purchaseOrders));
    }

    public function archive(Warehouse $warehouse)
    {
        // prevent archive the FBA warehouses
        if ($warehouse->integration_instance_id) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.archived_failed', ['id' => $warehouse->name]), 'ArchiveWarehouse'.Response::CODE_UNACCEPTABLE, 'id', ['id' => $warehouse->id]);
        }

        if ($warehouse->hasStockAvailable()) {
            return $this->response->error()
                ->addError(__('messages.warehouse.not_archived_has_stock', [
                    'resource' => $this->resource,
                    'id' => $warehouse->name,
                ]), 'Warehouse'.Response::CODE_HAS_STOCK, 'id', ['id' => $warehouse->id])
                ->addData(WarehouseResource::make($warehouse));
        }

        if ($warehouse->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $warehouse->name,
                ]))
                ->addData(WarehouseResource::make($warehouse));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $warehouse->name,
            ]), 'Warehouse'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $warehouse->id])
            ->addData(WarehouseResource::make($warehouse));
    }

    public function unarchived(Warehouse $warehouse)
    {
        // prevent archive the FBA warehouses
        if ($warehouse->integration_instance_id) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError(__('messages.warehouse.archived_failed', ['id' => $warehouse->name]), 'UnArchiveWarehouse'.Response::CODE_UNACCEPTABLE, 'id', ['id' => $warehouse->id]);
        }

        if ($warehouse->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $warehouse->name,
                ]))
                ->addData(WarehouseResource::make($warehouse));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $warehouse->name,
            ]), 'Warehouse'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $warehouse->id])
            ->addData(WarehouseResource::make($warehouse));
    }

    /**
     * Delete warehouse.
     *
     *
     * @throws Exception
     */
    public function destroy(Warehouse $warehouse): Response
    {
        $reasons = $warehouse->delete();

        // check if the nominalCode is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['warehouse_id' => $warehouse->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'warehouse',
                    'id' => $warehouse->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'warehouse',
            'id' => $warehouse->name,
        ]));
    }

    /**
     * Bulk delete warehouses.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:warehouses,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $warehouses = Warehouse::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($warehouses as $key => $warehouse) {
            $isUsed = $warehouse->isUsed();

            $result[$key] = $warehouse->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    public function isArchivable(Request $request)
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:warehouses,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $warehouses = Warehouse::with([])->whereIn('id', $ids)->select('id', 'name', 'integration_instance_id')->get();
        foreach ($warehouses as $key => $warehouse) {
            $isArchivable = $warehouse->isArchivable();

            $result[$key] = $warehouse->only('id', 'name');
            $result[$key]['archivable'] = $isArchivable === true;
            $result[$key]['reason'] = $isArchivable !== true ? $isArchivable : null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function archiveInstance(Warehouse $instance)
    {
        // prevent archive the FBA warehouses
        if ($instance->integration_instance_id) {
            $this->response->addWarning(__('messages.warehouse.archived_failed', ['id' => $instance->name]), 'ArchiveWarehouse'.Response::CODE_UNACCEPTABLE, "ids.{$instance->id}.id", ['id' => $instance->id]);

            return false;
        }

        return $this->originalArchiveInstance($instance);
    }

    /**
     * Get Automated Warehouses by shipping providers.
     */
    public function automatedWarehouses()
    {
        return $this->response->addData(Warehouse::getAutomatedWarehouses());
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return Warehouse::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return WarehouseResource::class;
    }
}
