<?php

namespace App\Http\Controllers\Traits;

use App\DTO\DataImporterDto;
use App\Exceptions\DataImportNoRecordsImportedException;
use App\Exceptions\DataImportTooLargeException;
use App\Exceptions\DataImportValidationErrorsException;
use App\Exceptions\ImportFailedException;
use App\Exporters\MapsExportableFields;
use App\Http\Requests\ImportRequest;
use App\Importers\ImportableInterface;
use App\Jobs\DataImportJob;
use App\Managers\DataImportManager;
use App\Response;
use Illuminate\Database\Eloquent\Model;
use League\Csv\SyntaxError;

/**
 * Trait ImportsData.
 *
 * @method string|Model getModel
 */
trait ImportsData
{
    /**
     * @return mixed
     *
     * @throws ImportFailedException
     */
    public function previewImport(ImportRequest $request)
    {
        if (! method_exists($this, 'getModel')) {
            return $this->sendImportErrorResponse();
        }

        if (! in_array(ImportableInterface::class, class_implements($this->getModel()))) {
            return $this->sendModelNotImportableResponse();
        }

        $importManager = (new DataImportManager(app($this->getModel()), DataImporterDto::from($request->validated())));

        try {
            $response = $this->sendPreviewResponse($importManager);
        } catch (SyntaxError $e) {
            return $this->response
                ->addError("CSV Import has invalid syntax: " . $e->getMessage(), Response::CODE_UNACCEPTABLE, "ImportData")
                ->setStatusCode(400);
        }

        return $response;
    }

    /**
     * @return mixed
     *
     * @throws ImportFailedException
     */
    public function import(ImportRequest $request)
    {
        $this->response = new Response();
        if (! method_exists($this, 'getModel')) {
            return $this->sendImportErrorResponse();
        }

        if (! in_array(ImportableInterface::class, class_implements($this->getModel()))) {
            return $this->sendModelNotImportableResponse();
        }

        $importManager = (new DataImportManager(app($this->getModel()), DataImporterDto::from($request->validated())));

        $requiredDifferences = $importManager->getRequiredFieldDifferences();

        if (count($requiredDifferences) > 0) {
            return $this->sendPreviewResponse($importManager);
        }

        try {
            // Main import function
            $importManager->import();
        } catch (SyntaxError $e) {
            return $this->response
                ->addError("Data import failed. {$e->getMessage()}", Response::CODE_UNACCEPTABLE, 'ImportData')
                ->setStatusCode(400);
        } catch (DataImportTooLargeException) {
            // For large imports, we deploy a job
            dispatch(new DataImportJob(auth()->user(), $this->getModel(), DataImporterDto::from($request->validated())))->onQueue('import-export');

            return $this->response->addData('The data will be imported shortly.');
        } catch (ImportFailedException $e) {
            return $this->sendImportErrorResponse();
        } catch (DataImportNoRecordsImportedException $e) {
            return $this->response
                ->addError('Data import is failed ,no records for import ', Response::CODE_UNACCEPTABLE, 'ImportData', $e->getValidationErrors())
                ->setStatusCode(500);

        } catch (DataImportValidationErrorsException $e) {
            return $this->response
                ->addError($e->getMessage(), Response::CODE_UNACCEPTABLE, 'ImportData', $e->getValidationErrors())
                ->setStatusCode(500);
        }

        return $this->response->addData('Data import was successful.');
    }

    /**
     * @return mixed
     *
     * @throws ImportFailedException
     */
    private function sendPreviewResponse(DataImportManager $importManager)
    {
        if (in_array(MapsExportableFields::class, class_implements($this->getModel()))) {
            $unImportable = collect($this->getModel()::getExportableFields())
                ->filter(function ($field) {
                    return $field['importable'] == false;
                })->map(function ($field) {
                    return $field['exported_as'];
                })->toArray();
        } else {
            $unImportable = [];
        }

        return $this->response->addData($importManager->getPreviewResponse($unImportable));
    }

    /**
     * @return mixed
     */
    private function sendImportErrorResponse()
    {
        return $this->response->error()
            ->addError(
                __('messages.failed.import_failed'),
                'Import'.Response::CODE_UNACCEPTABLE,
                class_basename($this->getModel())
            )
            ->setStatusCode(500);
    }

    /**
     * @return mixed
     */
    private function sendModelNotImportableResponse()
    {
        return $this->response->error()
            ->addError(__('messages.failed.model_not_importable', [
                'model' => class_basename($this->getModel()),
            ]), 'Import'.Response::CODE_UNACCEPTABLE, class_basename($this->getModel()))
            ->setStatusCode(500);
    }
}
