<?php

namespace App\Http\Controllers;

use App\DataTable\DataTableConfiguration;
use App\Http\Requests\StoreTableView;
use App\Http\Resources\TableViewResource;
use App\Models\TableView;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class TableViewController extends Controller
{
    private $resource = 'table view';

    public function index(Request $request): AnonymousResourceCollection
    {
        $tableViews = TableView::with([])->orderBy('is_default', 'DESC');
        $modelColumns = [];
        if (! empty($request->get('model'))) {
            $tableViews->where('model', $request->get('model'));

            $modelColumns = DataTableConfiguration::getTableSpecifications($request->get('model'))['table_specifications']['columns'];
            $modelColumns = collect($modelColumns)->pluck('data_name')->toArray();
        }

        $tableViews = $tableViews->get();
        $tableViews = TableView::getFiltered($tableViews, $modelColumns);
        // die(json_encode($tableViews, JSON_PRETTY_PRINT));

        return TableViewResource::collection($tableViews)->additional(['status' => __('messages.status_success')]);
    }

    public function store(StoreTableView $request): JsonResponse
    {
        $tableView = new TableView($request->all());
        $tableView->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData($tableView);
    }

    public function update(StoreTableView $request, $id): JsonResponse
    {
        $tableView = TableView::with([])->findOrFail($id);
        $tableView->fill($request->all());
        $tableView->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $tableView->name,
            ]))
            ->addData($tableView);
    }

    public function show($id): Response
    {
        return $this->response->addData(TableViewResource::make(TableView::with([])->findOrFail($id)));
    }

    /**
     * @throws Exception
     */
    public function destroy($id): JsonResponse
    {
        $tableView = TableView::with([])->findOrFail($id);
        $tableView->delete();

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $tableView->name,
        ]));
    }

    public function setDefaultView(int $viewId)
    {
        $tableView = TableView::with([])->findOrFail($viewId);

        $tableView->is_default = true;
        $tableView->is_favorite = true;
        $tableView->save();

        return $this->response->setMessage(__('messages.success.default', [
            'resource' => 'Table View',
            'id' => $tableView->name,
        ]))->addData($tableView);
    }
}
