<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/17/20
 * Time: 10:29 AM.
 */

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\SupplierInventoryRequest;
use App\Http\Resources\SupplierInventoryResource;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Repositories\SupplierInventoryRepository;
use App\Response;
use App\Services\SupplierInventory\SupplierInventoryCSVManager;
use Exception;
use Illuminate\Http\Request;

class SupplierInventoryController extends Controller
{
    use BulkOperation, ImportsData;
    use DataTable
  {
      export as exportTrait;
  }

    /**
     * @var string
     */
    protected $model_path = SupplierInventory::class;

    /**
     * @var string
     */
    protected $resource = 'supplier inventory';

    /**
     * @var SupplierInventoryRepository
     */
    private $inventories;

    /**
     * @var SupplierInventoryCSVManager
     */
    private $csvManager;

    /**
     * SupplierInventoryController constructor.
     */
    public function __construct(SupplierInventoryRepository $inventories, SupplierInventoryCSVManager $csvManager)
    {
        parent::__construct();
        $this->inventories = $inventories;
        $this->csvManager = $csvManager;
    }

    public function show($inventoryId): Response
    {
        $inventory = $this->inventories->find(e($inventoryId));
        $inventory->load(DataTableConfiguration::getRequiredRelations(SupplierInventory::class));

        return $this->response->addData(SupplierInventoryResource::make($inventory));
    }

    public function update(SupplierInventoryRequest $request, $inventoryId): Response
    {
        try {
            $payload = $request->validated();
            $payload['source'] = SupplierInventory::SOURCE_USER;
            $inventory = $this->inventories->updateInventory(e($inventoryId), $payload);
            $inventory->load(DataTableConfiguration::getRequiredRelations($this->model_path));

            return $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.update', ['resource' => $this->resource]))
                ->addData(SupplierInventoryResource::make($inventory));
        } catch (Exception $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setMessage($e->getMessage());
        }
    }

    public function export(Request $request, $supplierId)
    {
        // We bind the supplier id into the filters
        $filters = json_decode($request->query('filters'), true) ?? [
            'conjunction' => 'and',
            'filterSet' => [],
        ];

        $filters['filterSet'][] = [
            'column' => 'supplier_id',
            'operator' => '=',
            'value' => e($supplierId),
        ];

        $request->query->set('filters', json_encode($filters));

        return $this->exportTrait($request);
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    public function archive(SupplierInventory $inventory): Response
    {
        if ($inventory->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $inventory->id,
                ]))
                ->addData(SupplierInventoryResource::make($inventory));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $inventory->id,
            ]), 'SupplierInventory'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $inventory->id])
            ->addData(SupplierInventoryResource::make($inventory));
    }

    public function unarchived(SupplierInventory $inventory): Response
    {
        if ($inventory->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $inventory->id,
                ]))
                ->addData(SupplierInventoryResource::make($inventory));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $inventory->id,
            ]), 'SupplierInventory'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $inventory->id])
            ->addData(SupplierInventoryResource::make($inventory));
    }

    protected function getModel(): string
    {
        return SupplierInventory::class;
    }

    protected function getResource(): string
    {
        return SupplierInventoryResource::class;
    }
}
