<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreSupplier;
use App\Http\Resources\ProductSupplierStockResource;
use App\Http\Resources\SupplierResource;
use App\Models\Product;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Repositories\WarehouseRepository;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

/**
 * Class SupplierController.
 */
class SupplierController extends Controller
{
    use BulkOperation, DataTable, ImportsData;

    /**
     * @var string
     */
    protected $model_path = Supplier::class;

    /**
     * @var string
     */
    private $resource = 'supplier';

    /**
     * @var WarehouseRepository
     */
    private $warehouses;

    /**
     * SupplierController constructor.
     */
    public function __construct(WarehouseRepository $warehouses)
    {
        $this->warehouses = $warehouses;
        parent::__construct();
    }

    public function store(StoreSupplier $request): Response
    {
        $input = $request->validated();

        $supplier = new Supplier($input);
        $supplier->save();

        // Create address if available.
        if (isset($input['address'])) {
            $supplier->updateOrCreateAddress($input['address']);
        }

        // We create a default warehouse for the supplier
        $this->warehouses->createDefaultWarehouseForSupplier($supplier);

        // Attach pricing tiers
        $this->attachPricingTiers($supplier, $input);

        $supplier->load('address', 'pricingTiers', 'defaultPricingTier');

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(SupplierResource::make($supplier));
    }

    public function show(Supplier $supplier)
    {
        $supplier->load('warehouses');

        return $this->response->addData(SupplierResource::make($supplier));
    }

    public function update(StoreSupplier $request, $supplierId): Response
    {
        $inputs = $request->validated();

        $supplier = Supplier::with([])->findOrFail(e($supplierId));
        $supplier->fill($inputs);
        $supplier->save();

        // Update address if available
        if (isset($inputs['address'])) {
            $supplier->updateOrCreateAddress($inputs['address']);
        }

        // Attach pricing tiers if available.
        if (isset($inputs['pricing'])) {
            $tiers = array_map(function ($tier) {
                return $tier['id'];
            }, $inputs['pricing']);
            $supplier->attachPricingTiers($tiers);
        }

        $supplier->load('address', 'pricingTiers', 'defaultPricingTier', 'warehouses');

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $supplier->name,
            ]))
            ->addData(SupplierResource::make($supplier));
    }

    private function attachPricingTiers(Supplier $supplier, array $payload)
    {
        // Add in default pricing tiers.
        $tiers = SupplierPricingTier::with([])->where('is_default', true)->pluck('id')->toArray();

        if (isset($payload['pricing'])) {
            // Add other pricing tiers attached
            $ids = array_map(function ($tier) {
                return $tier['id'];
            }, $payload['pricing']);
            $tiers = array_unique(array_merge($tiers, $ids));
        }

        $supplier->attachPricingTiers($tiers);
    }

    /**
     * Delete supplier.
     *
     *
     * @throws Exception
     */
    public function destroy(Supplier $supplier): Response
    {
        $reasons = $supplier->delete();

        // check if the supplier is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['supplier_id' => $supplier->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'supplier',
                    'id' => $supplier->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'supplier',
            'id' => $supplier->name,
        ]));
    }

    public function archive(Supplier $supplier): Response
    {
        if ($supplier->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $supplier->name,
                ]))
                ->addData($supplier);
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => 'supplier',
                'id' => $supplier->name,
            ]), 'Supplier'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $supplier->id])
            ->addData($supplier);
    }

    public function unarchived(Supplier $supplier): Response
    {
        if ($supplier->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $supplier->name,
                ]))
                ->addData($supplier);
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'supplier',
                'id' => $supplier->name,
            ]), 'Supplier'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $supplier->id])
            ->addData($supplier);
    }

    /**
     * bulk delete using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:suppliers,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $suppliers = Supplier::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($suppliers as $key => $supplier) {
            $isUsed = $supplier->isUsed();

            $result[$key] = $supplier->only('id', 'code');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    public function supplierProductsStock(Request $request, Supplier $supplier)
    {
        $request->validate(
            [
                'ids.*' => 'exists:products,id',
            ]
        );
        $products = Product::with(['suppliersInventory' => function ($builder) use ($supplier) {
            $builder->where('supplier_id', $supplier->id);
        }])->whereIn('id', $request->ids)->get();

        $request->request->add(['supplier_id' => $supplier->id]);

        return ProductSupplierStockResource::collection($products);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return Supplier::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return SupplierResource::class;
    }
}
