<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\StoreEmailTemplate as StoreEmailTemplateRequest;
use App\Http\Resources\StoreEmailTemplateResource;
use App\Managers\StoreEmailTemplateManager;
use App\Models\StoreEmailTemplate;
use App\Repositories\StoreEmailTemplateRepository;
use App\Response;
use App\Services\EmailTemplate\EmailTemplatePreviewer;
use Exception;
use Illuminate\Http\Request;

class StoreEmailTemplateController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = StoreEmailTemplate::class;

    private $resource = 'StoreEmailTemplate';

    public function __construct(
        private StoreEmailTemplateRepository $emailTemplateRepository,
        private StoreEmailTemplateManager $storeEmailTemplateManager
    )
    {
        parent::__construct();
    }

    public function show($id): Response
    {
        $storeEmailTemplate = StoreEmailTemplate::with([])->findOrFail($id);

        return $this->response->addData(StoreEmailTemplateResource::make($storeEmailTemplate));
    }

    public function userTemplates(): Response
    {
        $storeEmailTemplates = $this->emailTemplateRepository->getNonSystemTemplates();

        return $this->response->addData(StoreEmailTemplateResource::collection($storeEmailTemplates));
    }

    public function store(StoreEmailTemplateRequest $request): Response
    {
        $storeEmailTemplate = new StoreEmailTemplate($request->all());
        $storeEmailTemplate->is_system_template = false;
        $storeEmailTemplate->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->response]))
            ->addData(StoreEmailTemplateResource::make($storeEmailTemplate));
    }

    public function update(StoreEmailTemplateRequest $request, $id): Response
    {
        $storeEmailTemplate = StoreEmailTemplate::with([])->findOrFail($id);
        $storeEmailTemplate->fill($request->all());
        $storeEmailTemplate->save();

        return $this->response->setMessage(__('messages.success.update', ['resource' => $this->response]))
            ->addData(StoreEmailTemplateResource::make($storeEmailTemplate));
    }

    public function send(Request $request): Response
    {
        $request->validate([
            'order_type' => 'required',
            'order_id' => 'required',
            'subject' => 'required',
            'html_body' => 'required'
        ]);

        $model = $this->storeEmailTemplateManager->getModelObjectFromType($request->order_type, $request->order_id);

        if(
            !$this->storeEmailTemplateManager->sendCustomerEmailTemplate(
                $model,
                $request->subject,
                $request->html_body,
                $request->get('cc', []) ?? []
            )
        ) {
            return $this->response->addError(
                'Order type not supported',
                Response::HTTP_BAD_REQUEST, ''
            )->error(Response::HTTP_BAD_REQUEST);
        }

        return $this->response->setMessage('Email sent successfully');
    }

    /**
     * @throws Exception
     */
    public function destroy($id): Response
    {
        $storeEmailTemplate = StoreEmailTemplate::with([])->findOrFail($id);
        $storeEmailTemplate->delete();

        return $this->response->setMessage(__('messages.success.delete', ['resource' => $this->response]));
    }

    /**
     * Bulk delete store email templates.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    public function archive($id): Response
    {
        $storeEmailTemplate = StoreEmailTemplate::with([])->findOrFail($id);

        if ($storeEmailTemplate->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                ]))
                ->addData(StoreEmailTemplateResource::make($storeEmailTemplate));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
            ]), 'Email Template '.Response::CODE_ALREADY_ARCHIVED, 'id', $storeEmailTemplate->id)
            ->addData(StoreEmailTemplateResource::make($storeEmailTemplate));
    }

    public function unarchived($id): Response
    {
        $storeEmailTemplate = StoreEmailTemplate::with([])->findOrFail($id);

        if ($storeEmailTemplate->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                ]))
                ->addData(StoreEmailTemplateResource::make($storeEmailTemplate));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_unarchived', [
                'resource' => $this->resource,
            ]), 'Email Template '.Response::CODE_ALREADY_UNARCHIVED, 'id', $storeEmailTemplate->id)
            ->addData(StoreEmailTemplateResource::make($storeEmailTemplate));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    public function preview(Request $request)
    {
        $isVariables = filter_var($request->input('variables', false), FILTER_VALIDATE_BOOLEAN);

        if (in_array($request->type, ['bulk_purchase_order'])) {
            return EmailTemplatePreviewer::bulkPreview($request->id, $request->type, $request->subject, $request->body, $isVariables);
        }

        return EmailTemplatePreviewer::fillTemplate($request->id, $request->type, $request->subject, $request->body, $isVariables);
    }

    protected function getModel(): string
    {
        return StoreEmailTemplate::class;
    }

    protected function getResource(): string
    {
        return StoreEmailTemplateResource::class;
    }
}
