<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreStore;
use App\Http\Resources\StoreResource;
use App\Models\Address;
use App\Models\Store;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class StoreController extends Controller
{
    use BulkOperation, DataTable, ImportsData;

    protected $model_path = Store::class;

    public function show(Store $store)
    {
        $store->load('address');

        return $this->response->addData(StoreResource::make($store));
    }

    /**
     * Create a new SCB.
     */
    public function store(StoreStore $request): Response
    {
        $inputs = $request->validated();

        $store = new Store($inputs);

        $store->address()->associate(Address::create($inputs));

        $store->save();

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'store']))
            ->addData(StoreResource::make($store));
    }

    /**
     * Update SCB.
     */
    public function update(StoreStore $request, Store $store): Response
    {
        $inputs = $request->validated();

        $store->fill($inputs);
        $store->setLogoUrl(array_key_exists('logo_url', $inputs) ? $inputs['logo_url'] : false, $request->input('download_logo', false));

        // save address
        if ($store->address) {
            $store->address->fill($inputs)->save();
        } else {
            $store->address()->associate(Address::create($inputs));
        }

        $store->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => 'store',
                'id' => $store->name,
            ]))
            ->addData(StoreResource::make($store));
    }

    /**
     * Archive the store.
     */
    public function archive(Store $store): Response
    {
        if ($store->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => 'store',
                    'id' => $store->name,
                ]))
                ->addData(StoreResource::make($store));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => 'store',
                'id' => $store->name,
            ]), 'Store'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $store->id])
            ->addData(StoreResource::make($store));
    }

    /**
     * Unarchived the store.
     */
    public function unarchived(Store $store): Response
    {
        if ($store->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'store',
                    'id' => $store->name,
                ]))
                ->addData(StoreResource::make($store));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'store',
                'id' => $store->name,
            ]), 'Store'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $store->id])
            ->addData(StoreResource::make($store));
    }

    /**
     * Delete store.
     *
     *
     * @throws Exception
     */
    public function destroy(Store $store): Response
    {
        $reasons = $store->delete();

        // check if the customer is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['store_id' => $store->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'store',
                    'id' => $store->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'store',
            'id' => $store->name,
        ]));
    }

    /**
     * Bulk delete nominal codes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:stores,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $stores = Store::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($stores as $key => $store) {
            $isUsed = $store->isUsed();

            $result[$key] = $store->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return Store::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return StoreResource::class;
    }
}
