<?php

namespace App\Http\Controllers;

use App\Data\CreateStockTakeData;
use App\Data\UpdateStockTakeData;
use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\CantDeleteStockTakeItemsForClosedStockTakeException;
use App\Exceptions\ClosedStockTakesCantChangeQuantityException;
use App\Exceptions\InsufficientInventoryStockException;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\NegativeAdjustmentOnStockTakeCannotBeModifiedException;
use App\Exceptions\NegativeInventoryStockTakeException;
use App\Helpers;
use App\Helpers\ExcelHelper;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\StockTakeRequest;
use App\Http\Resources\StockTakeItemResource;
use App\Http\Resources\StockTakeResource;
use App\Models\Product;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\Warehouse;
use App\Repositories\StockTakeRepository;
use App\Response;
use App\Services\StockTake\StockTakeManager;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use InvalidArgumentException;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

use Throwable;
use function __;
use function array_unique;

/**
 * Class StockTakeController.
 */
class StockTakeController extends Controller
{
    const STOCK_TAKE_RESOURCE_NAME = 'stock take';

    use BulkOperation, DataTable;

    private StockTakeManager $manager;

    private StockTakeRepository $stockTakes;

    protected string $model_path = StockTake::class;

    /**
     * StockTakeController constructor.
     */
    public function __construct(StockTakeManager $manager, StockTakeRepository $stockTakes)
    {
        parent::__construct();
        $this->manager = $manager;
        $this->stockTakes = $stockTakes;
    }

    public function show($stockTakeId): Response
    {
        $stockTake = StockTake::with(DataTableConfiguration::getRequiredRelations($this->getModel()))->findOrFail($stockTakeId);

        return $this->response->addData(StockTakeResource::make($stockTake));
    }

    /**
     * @throws Throwable
     */
    public function store(CreateStockTakeData $data): Response
    {
        try {
            $stockTake = $this->manager->createStockTake($data);
            $stockTake->load(DataTableConfiguration::getRequiredRelations($this->getModel()));

            return $this->response->addData(
                StockTakeResource::make($stockTake)
            );
        } catch (Exception $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setMessage($e->getMessage());
        }
    }

    /**
     * @throws Throwable
     */
    public function update(UpdateStockTakeData $data, StockTake $stockTake): Response
    {
        $this->response = new Response(); // To clear for testing

        $stockTake->load(['warehouse']);
        try {
            $stockTake = $this->manager->modifyStockTake($stockTake, $data);
        } catch (
            CantDeleteStockTakeItemsForClosedStockTakeException |
            ClosedStockTakesCantChangeQuantityException |
            NegativeAdjustmentOnStockTakeCannotBeModifiedException
            $e
        ) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError($e->getMessage(), 'StockTake', 'id', ['id' => $stockTake->id]);
        }

        return $this->response->addData(StockTakeResource::make($stockTake));
    }

    public function initiate(Request $request, StockTake $stockTake): Response
    {
        $this->validate($request, [
            'date_count' => 'nullable|date|before:tomorrow',
        ]);

        try {
            $stockTake = $this->manager->initiateCount($stockTake, $request->input('date_count'));

            return $this->response->addData(StockTakeResource::make($stockTake));
        } catch (NegativeInventoryStockTakeException $negativeInventoryException) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setErrors($negativeInventoryException->getResponseError());
        } catch (Exception $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setMessage($e->getMessage());
        }
    }

    public function finalize(StockTake $stockTake): Response
    {
        // TODO: Address with bulk once BulkInventoryManager has a method to deal with positive events as well as negative ones
        set_time_limit(0);
        try {
            $stockTake = $this->manager->finalizeStockTake($stockTake);

            return $this->response->addData(StockTakeResource::make($stockTake));
        } catch (InvalidArgumentException $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setMessage($e->getMessage());
        } catch (InsufficientStockException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(
                    __('messages.inventory.fifo_layer_not_existed', ['product_sku' => $e->productSku]),
                    Response::CODE_NO_ENOUGH_QUANTITY,
                    'product_sku'
                );
        } catch (InsufficientInventoryStockException $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setErrors([$e->getMessage()])
                ->setMessage($e->getMessage());
        }
    }

    public function revertToDraft($stockTakeId)
    {
        try {
            $stockTake = $this->manager->revertToDraft(e($stockTakeId));

            return $this->response->addData(StockTakeResource::make($stockTake));
        } catch (InvalidArgumentException $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setErrors([$e->getMessage()])
                ->setMessage($e->getMessage());
        }
    }

    public function archive(StockTake $stockTake)
    {
        if ($stockTake->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => self::STOCK_TAKE_RESOURCE_NAME,
                    'id' => $stockTake->id,
                ]))
                ->addData(StockTakeResource::make($stockTake));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => self::STOCK_TAKE_RESOURCE_NAME,
                'id' => $stockTake->id,
            ]), 'Warehouse'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $stockTake->id])
            ->addData(StockTakeResource::make($stockTake));
    }

    public function unarchived(StockTake $stockTake)
    {
        if ($stockTake->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => self::STOCK_TAKE_RESOURCE_NAME,
                    'id' => $stockTake->id,
                ]))
                ->addData(StockTakeResource::make($stockTake));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => self::STOCK_TAKE_RESOURCE_NAME,
                'id' => $stockTake->id,
            ]), 'Warehouse'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $stockTake->id])
            ->addData(StockTakeResource::make($stockTake));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Delete stock take.
     *
     *
     * @throws Exception
     */
    public function destroy(StockTake $stockTake): Response
    {
        // Delete the stock take
        $reasons = $stockTake->delete();

        // check if the inventory adjustment has fifo-layer uncovered fulfilled quantity
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['stock_take_id' => $stockTake->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'stock take',
                    'id' => $stockTake->id,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => self::STOCK_TAKE_RESOURCE_NAME,
            'id' => $stockTake->id,
        ]));
    }

    public function isDeletable(Request $request)
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:stock_takes,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $stockTakes = StockTake::with([])->whereIn('id', $ids)->get();
        foreach ($stockTakes as $key => $stockTake) {
            $isUsed = $stockTake->isUsed();

            $result[$key] = $stockTake->only('id');
            $result[$key]['deletable'] = $isUsed ? false : true;
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * Bulk delete stock takes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * Bulk upload lines.
     */
    public function importStockTakeLines(Request $request): Response
    {
        $request->validate(['file' => 'required', 'separator' => 'nullable', 'escape' => 'nullable']);

        $filePath = storage_path(config('uploader.models.target').$request->input('file'));
        $separator = $request->input('separator', ',');
        $escape = $request->input('escape', '"');

        // check header
        $header = array_keys(Helpers::csvFileToArray($filePath, $separator, $escape)->current());
        if (count(array_intersect($header, ['quantity'])) !== 1 && count(array_intersect($header, ['sku', 'id'])) === 0) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError("doesn't match the template columns", 'TemplateNotMatched', 'file');
        }

        // add lines
        $lines = collect();
        foreach (Helpers::csvFileToArray($filePath, $separator, $escape) as $line) {
            $stockTakeLine = new StockTakeItem();
            $stockTakeLine->product_id = empty($line['id']) ? null : $line['id'];
            $stockTakeLine->sku = empty($line['sku']) ? null : $line['sku'];
            $stockTakeLine->qty_counted = in_array('qty_counted', $header) ? ($line['qty_counted'] == '' ? 1 : $line['qty_counted']) : 0;
            $stockTakeLine->snapshot_inventory = in_array('snapshot_inventory', $header) ? ($line['snapshot_inventory'] == '' ? 0 : $line['snapshot_inventory']) : 0;
            $stockTakeLine->unit_cost = in_array('unit_cost', $header) ? ($line['unit_cost'] == '' ? null : $line['unit_cost']) : null;
            $lines->add($stockTakeLine);
        }
        // match skus with products
        $lines->chunk(100)->map(function (Collection $chunk) {
            $skus = $chunk->pluck('sku')->filter()->unique();
            $ids = $chunk->pluck('product_id')->filter()->unique();
            $products = Product::with(['primaryImage'])->whereIn('sku', $skus)->orWhereIn('id', $ids)->get(['id', 'sku', 'barcode', 'name']);

            return $chunk->map(function (StockTakeItem $stockTakeItem) use ($products) {
                if ($stockTakeItem->product_id) {
                    return $stockTakeItem->setRelation('product', $products->firstWhere('id', $stockTakeItem->product_id));
                }

                return $stockTakeItem->setRelation('product', $products->firstWhere('sku', $stockTakeItem->sku));
            });
        })->collapse();
        // unmatched lines

        $unmatchedLines = $lines->where('product', null)->pluck('sku');

        if ($unmatchedLinesCount = $lines->where('product', null)->count()) {
            if ($unmatchedLinesCount == $lines->count()) {
                return $this->response->error(Response::HTTP_BAD_REQUEST)->addError(__('messages.stock_take.imported_lines_unmatched', ['count' => 'All']), Response::CODE_HAS_UNMAPPED_PRODUCTS, 'sku');
            }
            $this->response->addWarning(__('messages.stock_take.imported_lines_unmatched', ['count' => "$unmatchedLinesCount of"]), Response::CODE_HAS_UNMAPPED_PRODUCTS, 'sku', $unmatchedLines->toArray());
        }
        /** @see SKU-3743 combine repeated lines that have the same product */
        $matchedLines = $lines->where('product', '!=', null)->groupBy('product.id')->map(function (Collection $productGroup) {
            /** @var StockTakeItem $firstLine */
            $firstLine = $productGroup->first();
            $firstLine->qty_counted = $productGroup->sum('qty_counted');

            return $firstLine;
        })->values();

        return $this->response->addData(StockTakeItemResource::collection($matchedLines));
    }

    /**
     * Download importing lines template.
     */
    public function exportStockTakeLines($stockTakeId): BinaryFileResponse
    {
        $stockTake = StockTake::with([])->findOrFail($stockTakeId);
        $lines = collect();
        StockTakeItem::with(['stockTake', 'product', 'productInventories'])->where('stock_take_id', $stockTakeId)->each(function (StockTakeItem $stockTakeItem) use ($lines) {
            $currentStock = $stockTakeItem->product_inventory->inventory_total ?? 0;
            $line = [
                'id' => $stockTakeItem->product_id,
                'sku' => $stockTakeItem->product->sku,
                'barcode' => $stockTakeItem->product->barcode,
                'name' => $stockTakeItem->product->name,
            ];
            if ($stockTakeItem->stockTake->status !== StockTake::STOCK_TAKE_STATUS_DRAFT) {
                $line['snapshot_inventory'] = $stockTakeItem->snapshot_inventory;
                $line['qty_counted'] = $stockTakeItem->qty_counted;
                $line['adjustment'] = $stockTakeItem->qty_counted - $stockTakeItem->snapshot_inventory;
                $line['unit_cost'] = $stockTakeItem->unit_cost ?? $stockTakeItem->product->unit_cost;
            }
            $lines->add($line);
        });
        $headers = ['id', 'sku', 'barcode', 'name']; // static to return them when lines are empty
        if ($stockTake->status !== StockTake::STOCK_TAKE_STATUS_DRAFT) {
            $headers[] = 'snapshot_inventory';
            $headers[] = 'qty_counted';
            $headers[] = 'adjustment';
            $headers[] = 'unit_cost';
        }
        $exportedFile = ExcelHelper::array2csvfile(
            'stock_take_lines.csv',
            $headers,
            $lines
        );

        return response()->download($exportedFile)->deleteFileAfterSend(true);
    }

    /**
     *  Bulk insert stock take items
     * @throws Throwable
     */
    public function bulkInsert(Request $request): Response
    {
        // validate
        $request->validate([
            'filters' => 'required_without:ids',
            'ids' => 'required_without:filters|array|min:1',
            'ids.*' => 'integer|exists:'.(new Product())->getTable().',id',
            'warehouse_id' => 'integer|exists:'.(new Warehouse())->getTable().',id',
        ]);

        // Create stock take
        $stockTake = $this->manager->createStockTake(CreateStockTakeData::from([
            'warehouse_id' => $request->input('warehouse_id'),
            'status' => StockTake::STOCK_TAKE_STATUS_DRAFT,
        ]));

        // Create insert array
        $productIds = array_unique($request->input('ids'));
        $insertArr = [];
        foreach ($productIds as $productId) {
            $insertArr[] = [
                'stock_take_id' => $stockTake->id,
                'product_id' => $productId,
            ];
        }

        // Bulk insert
        StockTakeItem::insert($insertArr);

        return $this->response->setMessage(__('messages.success.create', [
            'resource' => self::STOCK_TAKE_RESOURCE_NAME,
            'id' => $stockTake->id,
        ]))->addData($stockTake->id, 'stock_take_id');
    }

    protected function getModel(): string
    {
        return StockTake::class;
    }

    protected function getResource(): string
    {
        return StockTakeResource::class;
    }
}
