<?php

namespace App\Http\Controllers\Starshipit;

use App\Http\Controllers\IntegrationInstanceController;
use App\Integrations\Starshipit;
use App\Jobs\Starshipit\ProcessStarshipitWebhookJob;
use App\Jobs\Starshipit\SubmitOrders;
use App\Models\IntegrationInstance;
use App\Models\SalesOrderFulfillment;
use App\Models\Starshipit\StarshipitOrder;
use App\Response;
use App\SDKs\Starshipit\StarshipitException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * Class OrderController.
 */
class OrderController extends IntegrationInstanceController
{
    /**
     * OrderController constructor.
     */
    public function __construct()
    {
        parent::__construct();
    }

    public function submitOrders(IntegrationInstance $integrationInstance): Response
    {
        SubmitOrders::dispatch($integrationInstance);

        return $this->response->setMessage('Added to the Queue, it will be process shortly');
    }

    /**
     * Update Starshipit order from Sales order fulfillment.
     *
     *
     * @throws StarshipitException
     */
    public function updateOrderFromFulfillment(SalesOrderFulfillment $salesOrderFulfillment): Response
    {
        $starshipitIntegrationInstance = IntegrationInstance::with([])->starshipit()->first();
        if (! $starshipitIntegrationInstance) {
            return $this->response->addError(
                __('messages.integration_instance.not_integrated', ['integration' => 'Starshipit']),
                'Starshipit'.Response::CODE_NOT_INTEGRATED,
                'starshipit'
            );
        }

        $validationErrors = $salesOrderFulfillment->toStarshipitOrder()->validate();
        if (! empty($validationErrors)) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.integration_instance.can_not_submit_order', ['resource' => 'Starshipit']))->addErrorsFromValidator($validationErrors);
        }

        $starshipit = new Starshipit($starshipitIntegrationInstance);

        $starshipitOrder = StarshipitOrder::with([])->firstOrNew(['sku_fulfillment_id' => $salesOrderFulfillment->id]);

        if ($starshipitOrder->exists && $starshipitOrder->order_id) {
            $submitOrderResponse = $starshipit->updateOrder($salesOrderFulfillment->toStarshipitOrder($starshipitOrder));
        } else {
            $submitOrderResponse = $starshipit->submitOrder($salesOrderFulfillment->toStarshipitOrder());
        }

        if ($submitOrderResponse->statusCode == 200 && ($submitOrderResponse->body['success'] ?? true)) {
            $starshipitOrder->json_object = $submitOrderResponse->body['order'];

            $starshipitOrder->errors = null;
            $starshipitOrder->save();
        } else {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError(
                __('messages.failed.update_api', ['resource' => 'Starshipit Order']),
                Response::CODE_UNEXPECTED,
                'sales_order_fulfillment_id',
                (array) ($submitOrderResponse->body['errors'] ?? $submitOrderResponse->body)
            );
        }

        return $this->response->setMessage(
            __('messages.success.update', ['resource' => 'Starshipit Order', 'id' => $starshipitOrder->order_number])
        );
    }

    public function viewOrder($salesOrderFulfillmentId)
    {
        $starshipitOrder = StarshipitOrder::with([])->where('sku_fulfillment_id', (int) $salesOrderFulfillmentId)->first();

        if ($starshipitOrder) {
            return $this->response->addData($starshipitOrder);
        }

        return $this->response->error(Response::HTTP_NOT_FOUND)->addError(__('messages.failed.not_found', ['resource' => 'Starshipit details']), Response::CODE_NOT_FOUND, 'id');
    }

    public function receiveTrackingByWebhook(Request $request): JsonResponse
    {
        if (empty($request->all())) {
            return response()->json(['status' => 'success']);
        }
        dispatch(new ProcessStarshipitWebhookJob($request->all()))->onQueue('fulfillments');

        return response()->json(['status' => 'success']);
    }
}
