<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreShippingProvider;
use App\Http\Resources\ShippingMethodMappingShippingProviderResource;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\SalesOrderFulfillment;
use App\Models\ShippingMethod;
use App\Models\ShippingMethodMappingsSkuToShippingProviderMethod;
use App\Models\ShippingProvider;
use App\Response;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class ShippingProviderController extends Controller
{
    /**
     * Retrieve all shipping providers.
     */
    public function index(): JsonResponse
    {
        return $this->response->addData(ShippingProvider::all());
    }

    /**
     * Show a shipping provider.
     */
    public function show(ShippingProvider $shippingProvider): JsonResponse
    {
        return $this->response->addData($shippingProvider);
    }

    /**
     * Create a new shipping provider.
     */
    public function store(StoreShippingProvider $request): JsonResponse
    {
        $shippingProvider = new ShippingProvider($request->all());
        $shippingProvider->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => 'shipping provider']))
            ->addData($shippingProvider);
    }

    /**
     * Update a shipping provider.
     */
    public function update(StoreShippingProvider $request, $id): JsonResponse
    {
        $shippingProvider = ShippingProvider::with([])->findOrFail($id);
        $shippingProvider->fill($request->all());
        $shippingProvider->save();

        return $this->response->setMessage(__('messages.success.update', ['resource' => 'shipping provider']))
            ->addData($shippingProvider);
    }

    /**
     * Retrieve list of shipping provider's methods mapping.
     *
     * @param  IntegrationInstance  $integrationInstance as shipping provider
     * @return AnonymousResourceCollection|Response
     */
    public function shippingMethods(IntegrationInstance $integrationInstance)
    {
        if ($integrationInstance->integration->integration_type != Integration::TYPE_SHIPPING_PROVIDER) {
            return $this->response->addError(__('messages.integration_instance.not_shipping_provider'), Response::CODE_UNACCEPTABLE, 'id');
        }

        $shippingProviderMappings = ShippingMethodMappingsSkuToShippingProviderMethod::with(['shippingMethod'])->where('shipping_provider_id', $integrationInstance->id)->get();

        return ShippingMethodMappingShippingProviderResource::collectionWithTableSpecifications($shippingProviderMappings, null);
    }

    /**
     * Get sales orders count that used shipping method.
     */
    public function shippingMethodSalesOrdersCount(Request $request, IntegrationInstance $integrationInstance): Response
    {
        $request->validate(['shipping_method' => 'required']);

        switch ($integrationInstance->integration->name) {
            case Integration::NAME_SHIPSTATION:
                $fulfillmentType = SalesOrderFulfillment::TYPE_SHIPSTATION;
                break;
            case Integration::NAME_STARSHIPIT:
                $fulfillmentType = SalesOrderFulfillment::TYPE_STARSHIPIT;
                break;
            default:
                return $this->response->addError(__('messages.integration_instance.not_shipping_provider'), Response::CODE_UNACCEPTABLE, 'id');
        }

        $salesOrderFulfillmentsCount = SalesOrderFulfillment::with([])
            ->where('fulfillment_type', $fulfillmentType)
            ->where('fulfilled_shipping_method', $request->input('shipping_method'))
            ->count();

        return $this->response->setMessage(__('messages.sales_order.mapping_shipping_methods_count', [
            'salesOrdersCount' => $salesOrderFulfillmentsCount,
            'service_name' => $request->input('shipping_method'),
        ]));
    }

    /**
     * Map sales channel's shipping methods.
     *
     * @param  IntegrationInstance  $integrationInstance as shipping provider
     */
    public function mapShippingMethods(Request $request, IntegrationInstance $integrationInstance): Response
    {
        switch ($integrationInstance->integration->name) {
            case Integration::NAME_SHIPSTATION:
                $fulfillmentType = SalesOrderFulfillment::TYPE_SHIPSTATION;
                break;
            case Integration::NAME_STARSHIPIT:
                $fulfillmentType = SalesOrderFulfillment::TYPE_STARSHIPIT;
                break;
            default:
                return $this->response->addError(__('messages.integration_instance.not_shipping_provider'), Response::CODE_UNACCEPTABLE, 'id');
        }

        $request->validate([
            'mappings' => 'required|array|min:1',
            'mappings.*.id' => 'required_without:mappings.*.shipping_provider_method|exists:shipping_method_mappings_sku_to_shipping_provider_method,id',
            'mappings.*.shipping_provider_method' => 'required_without:mappings.*.id',
            'mappings.*.shipping_method_id' => 'nullable|exists:shipping_methods,id',
        ]);

        foreach ($request->input('mappings') as $mapping) {
            // save mapping
            if (isset($mapping['id'])) {
                $salesChannelMapping = ShippingMethodMappingsSkuToShippingProviderMethod::with([])->findOrFail($mapping['id']);
            } else {
                $shippingMethod = ShippingMethod::with([])->findOrFail($mapping['shipping_method_id']);
                $salesChannelMapping = ShippingMethodMappingsSkuToShippingProviderMethod::with([])
                    ->whereFullName($mapping['shipping_provider_method'])
                    ->firstOrCreate([
                        'shipping_provider_method' => $mapping['shipping_provider_method'],
                        'shipping_method_id' => $mapping['shipping_method_id'],
                        'shipping_provider_id' => $integrationInstance->id,
                        'shipping_provider_carrier' => $shippingMethod->shippingCarrier?->name ?? null,
                    ]);
            }
            $salesChannelMapping->shipping_method_id = $mapping['shipping_method_id'] ?? null;
            $salesChannelMapping->save();

            // reflect it on sales order fulfillments
            SalesOrderFulfillment::with([])->where('fulfillment_type', $fulfillmentType)
                ->where('fulfilled_shipping_method', $salesChannelMapping->full_name)
                ->update(['fulfilled_shipping_method_id' => $salesChannelMapping->shipping_method_id]);
        }

        return $this->response->addData(__('messages.success.create', ['resource' => 'shipping method mappings']));
    }

    /**
     * Delete the shipping mapping
     */
    public function destroyShippingMapping($shippingId): Response
    {
        $mapping = ShippingMethodMappingsSkuToShippingProviderMethod::query()->findOrFail($shippingId);
        $mapping->delete();

        return $this->response->setMessage(__('messages.success.delete', ['resource' => 'shipping method mapping', 'id' => $mapping->shipping_provider_method]));
    }
}
