<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreShippingMethod;
use App\Http\Resources\ShippingMethodResource;
use App\Models\ShippingMethod;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ShippingMethodController extends Controller
{
    use bulkOperation, DataTable, ImportsData;

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreShippingMethod $request): Response
    {
        $shippingMethod = new ShippingMethod($request->validated());
        $shippingMethod->save();

        $shippingMethod->load('shippingCarrier');

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'shipping method']))
            ->addData(ShippingMethodResource::make($shippingMethod));
    }

    /**
     * Display the specified resource.
     */
    public function show(ShippingMethod $shippingMethod): Response
    {
        $shippingMethod->load('shippingCarrier');

        return $this->response->addData(ShippingMethodResource::make($shippingMethod));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(StoreShippingMethod $request, ShippingMethod $shippingMethod): Response
    {
        $shippingMethod->update($request->validated());

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => 'shipping method',
                'id' => $shippingMethod->name,
            ]))
            ->addData(ShippingMethodResource::make($shippingMethod));
    }

    /**
     * Delete nominal code.
     *
     *
     * @throws Exception
     */
    public function destroy(ShippingMethod $shippingMethod): Response
    {
        $reasons = $shippingMethod->delete();

        // check if the shipping method is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['shipping_method_id' => $shippingMethod->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'shipping method',
                    'id' => $shippingMethod->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'shipping method',
            'id' => $shippingMethod->name,
        ]));
    }

    /**
     * Bulk delete nominal codes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * Archive the shipping method.
     */
    public function archive(ShippingMethod $shippingMethod): Response
    {
        if ($shippingMethod->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => 'shipping method',
                    'id' => $shippingMethod->name,
                ]))
                ->addData(ShippingMethodResource::make($shippingMethod));
        }

        return $this->response
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => 'shipping method',
                'id' => $shippingMethod->name,
            ]), 'ShippingMethod'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $shippingMethod->id])
            ->addData(ShippingMethodResource::make($shippingMethod));
    }

    /**
     * Unarchived the shipping method.
     */
    public function unarchived(ShippingMethod $shippingMethod): Response
    {
        if ($shippingMethod->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'shipping method',
                    'id' => $shippingMethod->name,
                ]))
                ->addData(ShippingMethodResource::make($shippingMethod));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'shipping method',
                'id' => $shippingMethod->name,
            ]), 'ShippingMethod'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $shippingMethod->id])
            ->addData(ShippingMethodResource::make($shippingMethod));
    }

    /**
     * Bulk archive nominal codes.
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * Bulk unarchived nominal codes.
     *
     *
     * @return mixed
     */
    public function bulkUnArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:shipping_methods,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $shippingMethods = ShippingMethod::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($shippingMethods as $key => $shippingMethod) {
            $isUsed = $shippingMethod->isUsed();

            $result[$key] = $shippingMethod->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    protected function getModel()
    {
        return ShippingMethod::class;
    }

    protected function getResource()
    {
        return ShippingMethodResource::class;
    }
}
