<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreShippingCarrier;
use App\Models\ShippingCarrier;
use App\Response;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Throwable;

class ShippingCarrierController extends Controller
{
    /**
     * Retrieve all shipping carriers.
     */
    public function index(): JsonResponse
    {
        return $this->response->addData(ShippingCarrier::with(['shippingMethods'])->get());
    }

    /**
     * View a Shipping Carrier.
     */
    public function show($id): JsonResponse
    {
        return $this->response->addData(ShippingCarrier::with(['shippingMethods'])->findOrFail($id));
    }

    /**
     * Create a new shipping carrier.
     *
     *
     * @throws Throwable
     */
    public function store(StoreShippingCarrier $request): JsonResponse
    {
        $inputs = $request->validated();

        DB::transaction(function () use ($inputs) {
            $shippingCarrier = new ShippingCarrier($inputs);
            $shippingCarrier->save();

            $shippingCarrier->shippingMethods()->createMany($inputs['shipping_methods'] ?? []);

            $shippingCarrier->load('shippingMethods');

            $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.create', ['resource' => 'shipping carrier']))
                ->addData($shippingCarrier);
        });

        return $this->response;
    }

    /**
     * Update a shipping carrier.
     *
     *
     * @throws Throwable
     */
    public function update(StoreShippingCarrier $request, $id): JsonResponse
    {
        $inputs = $request->validated();

        DB::transaction(function () use ($id, $inputs) {
            $shippingCarrier = ShippingCarrier::with([])->findOrFail($id);
            $shippingCarrier->update($inputs);

            foreach ($inputs['shipping_methods'] ?? [] as $method) {
                if (isset($method['id'])) {
                    $shippingMethod = $shippingCarrier->shippingMethods()->findOrFail($method['id']);
                    $shippingMethod->update($method);
                } else {
                    $shippingCarrier->shippingMethods()->updateOrCreate([
                        'name' => $method['name'],
                        'shipping_carrier_id' => $id,
                    ], $method);
                }
            }

            $shippingCarrier->load('shippingMethods');

            $this->response->setMessage(__('messages.success.update', ['resource' => 'shipping carrier']))
                ->addData($shippingCarrier);
        });

        return $this->response;
    }
}
