<?php

namespace App\Http\Controllers;

use App\Http\Requests\SalesOrderPaymentRequest;
use App\Http\Resources\PaymentResource;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Response;
use Carbon\Carbon;
use Illuminate\Http\Request;

class SalesOrderPaymentController extends Controller
{
    public function index($salesOrderId): Response
    {
        $salesOrder = SalesOrder::with(['payments'])->findOrFail(e($salesOrderId));

        return $this->response->addData(PaymentResource::collection($salesOrder->payments))->success(
            Response::HTTP_OK
        );
    }

    public function store(SalesOrderPaymentRequest $request, $salesOrderId): Response
    {
        $salesOrder = $this->findOrFailSalesOrder($salesOrderId);
        $data = $request->validated();

        if (empty($data['payment_date'])) {
            $data['payment_date'] = Carbon::now()->format('Y-m-d');
        }
        // Checked if the payment amount  more than the sales order balance ,
        // Then returned overpayment amount if it exist
        if ($overPaymentAmount = $this->checkIsOverPayment($data, $salesOrder)) {
            // Add a sales credit with an is_for_overpayment flag and with overpayment amount
            $salesCredit = new SalesCredit(['sales_order_id' => $salesOrder->id, 'total_credit' => $overPaymentAmount, 'is_for_overpayment' => 1]);
            $salesOrder->addSalesCredit($salesCredit);
            // Add  payment associated to the sales credit with overpayment amount
            $salesCredit->payments()->create(array_merge($data, ['currency_id' => $data['currency_id'] ?? $salesOrder->currency_id, 'amount' => $overPaymentAmount, 'type' => \App\Enums\PaymentType::TYPE_OVERPAYMENT]));
        }
        // Check this payment type is refund
        if ($data['amount'] < 0) {
            // Add sales credit associated with sales order
            $salesCredit = new SalesCredit(['sales_order_id' => $salesOrder->id, 'total_credit' => abs($data['amount'])]);
            $salesOrder->addSalesCredit($salesCredit);
            //  Add  payment associated to the sales credit with positive  amount
            $payment = $salesCredit->payments()->create(array_merge($data, ['currency_id' => $data['currency_id'] ?? $salesOrder->currency_id, 'amount' => abs($data['amount'])]));
        } else {
            // Add the payment to the sales order
            /** @var Payment $payment */
            $payment = $salesOrder->payments()->create(array_merge($data, ['currency_id' => $data['currency_id'] ?? $salesOrder->currency_id]));
        }

        return $this->paymentSaved($salesOrder, $payment, $data['payment_date'], 'create');
    }

    public function update(SalesOrderPaymentRequest $request, $salesOrderId, $paymentId): Response
    {
        $salesOrder = $this->findOrFailSalesOrder($salesOrderId);

        /** @var Payment $payment */
        $payment = $salesOrder->payments()->findOrFail(e($paymentId));

        $data = $request->validated();

        // Credit payment types cannot be changed as they are system created
        if ($payment->paymentType->name === PaymentType::SALES_CREDIT_PAYMENT_TYPE_NAME) {
            unset($data['payment_type_id']);
        }
        $payment->fill($data);
        $payment->save();

        return $this->paymentSaved($salesOrder, $payment, $data['payment_date'] ?? null, 'update');
    }

    /**
     * @param  null  $paymentDate
     */
    private function paymentSaved(SalesOrder $salesOrder, Payment $payment, $paymentDate, string $action): Response
    {
        // We update the payment status of the sales order
        $salesOrder->setPaymentStatus($paymentDate);

        if ($action !== strtolower(Request::METHOD_DELETE)) {
            $this->response->addData($payment);
        }

        return $this->response->setMessage(__("messages.success.{$action}", [
            'resource' => 'sales order payment',
            'id' => $payment->id,
        ]));
    }

    public function destroy($salesOrderId, $paymentId)
    {
        $salesOrder = $this->findOrFailSalesOrder(e($salesOrderId));

        /** @var Payment $payment */
        $payment = $salesOrder->payments()->findOrFail(e($paymentId));

        // Delete the payment
        $payment->delete();

        return $this->paymentSaved($salesOrder, $payment, null, 'delete');
    }

    private function findOrFailSalesOrder($id, array $relations = ['salesOrderLines']): SalesOrder
    {
        return SalesOrder::with($relations)->findOrFail(e($id));
    }

    private function checkIsOverPayment(mixed $attributes, SalesOrder $salesOrder): float
    {
        $balance = (float) (($salesOrder->calculated_total + ($salesOrder->exact_discount_total ? $salesOrder->exact_discount_total : 0)) + ($salesOrder->tax_total && ! $salesOrder->is_tax_included ? $salesOrder->tax_total : 0) - $salesOrder->total_paid);
        $isOverPayment = isset($attributes['amount']) && (float) $attributes['amount'] > 0 && (float) $attributes['amount'] > $balance;

        return $isOverPayment ? ((float) $attributes['amount']) - $balance : 0;
    }
}
