<?php

namespace App\Http\Controllers;

use App\DTO\ProductMappingDto;
use App\Events\SalesOrderLineDeleted;
use App\Helpers;
use App\Helpers\ExcelHelper;
use App\Http\Requests\SalesOrderLineReservationRequest;
use App\Http\Requests\SalesOrderRequest;
use App\Http\Resources\SalesOrderLineDetailResource;
use App\Http\Resources\SalesOrderLineResource;
use App\Http\Resources\SalesOrderResource;
use App\Jobs\MapSalesOrderLinesJob;
use App\Models\FinancialLineType;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\ProductPricing;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Repositories\WarehouseRepository;
use App\Response;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\StockTake\OpenStockTakeException;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Response as ResponseAlias;
use Throwable;

class SalesOrderLineController extends SalesOrderController
{
    /*
     * This method is used to reserve and unreserve for a given sales order line.
     */
    /**
     * @throws Throwable
     */
    public function reservations(SalesOrderLineReservationRequest $request): Response
    {
        try {
            $salesOrder = DB::transaction(function () use ($request) {
                return $this->manager->lineReservation(
                    $this->orders->findById($request->validated('sales_order_id')),
                    $request->validated()
                )
                ->load($this->getRequiredRelations());
            });

            return $this->response->success(ResponseAlias::HTTP_CREATED)
                ->setMessage(__('messages.success.create', ['resource' => 'sales order']))
                ->addData(SalesOrderResource::make($salesOrder));
        } catch (OpenStockTakeException $openStockTakeException) {
            return $this->response->error($openStockTakeException->status)
                ->setMessage($openStockTakeException->getMessage())
                ->setErrors(['resource' => 'stock take', 'id' => $openStockTakeException->stockTake->id]);
        }
    }

    public function show($id): Response
    {
        $salesOrderLine = SalesOrderLine::with(
            'backorderQueue.backorderQueueCoverages.purchaseOrderLine.purchaseOrder',
            'backorderQueue.backorderQueueReleases',
            'backorderQueue.supplier'
        )
            ->findOrFail($id);
        return $this->response->success()
            ->addData(SalesOrderLineDetailResource::make($salesOrderLine));
    }

    /**
     * Delete purchase order line.
     *
     * @throws Exception
     */
    public function destroyLine($line_id): JsonResponse
    {
        $salesOrderLine = SalesOrderLine::with([])->withCount('salesOrderFulfillmentLines')->findOrFail($line_id);

        /**
         * Check if Invoice issued for the purchase order.
         */
        if ($salesOrderLine->sales_order_fulfillment_lines) {
            return response()->json([
                'status' => 'failed',
                'message' => 'Sales order line fully/partially fulfilled. You can\'t delete it.',
            ], 422);
        }

        /**
         * Check if last line of purchase order lines.
         */
        if (SalesOrderLine::where('sales_order_id', $salesOrderLine->sales_order_id)->count() == 1) {
            return response()->json([
                'status' => 'failed',
                'message' => 'Can\'t delete last line of the sales order.',
            ], 422);
        }

        /**
         * Delete line.
         */
        $salesOrderLine->delete();

        /**
         * fire event and log activity.
         */
        event(new SalesOrderLineDeleted($salesOrderLine));

        return response()->json(['status' => 'success']);
    }

    /**
     * Mark sales order line as a non-product line.
     */
    public function markAsNonProduct(SalesOrderLine $salesOrderLine): Response
    {
        if ($salesOrderLine->product_id) {
            $this->response->error(Response::HTTP_BAD_REQUEST)->addError(__('messages.sales_order.line_already_mapped'), 'SalesOrderLine'.Response::CODE_ALREADY_MAPPED, 'id');
        }

        $salesOrderLine->is_product = false;
        $salesOrderLine->save();

        return $this->response->setMessage(__('messages.sales_order.line_marked_as_non_product'));
    }

    /**
     * Map the sales order line to a product
     *
     *
     * @throws Throwable
     */
    public function mapToProduct(Request $request, SalesOrderLine $salesOrderLine): Response
    {
        $request->validate(['product_id' => 'required|exists:products,id']);

        $salesOrderLine->product_id = $request->input('product_id');
        app(SalesOrderManager::class)->mapSalesOrderLine($salesOrderLine);

        $integrationInstance = $salesOrderLine->salesOrder->salesChannel->integrationInstance;

        /*
         * TODO: If listings count is > 1, we shouldn't try to universally map the line.  This whole process needs to be refactored
         *  on the mapping dialog, we shouldn't query products, we should query sales channel products. We need a way to get listings
         *  in the first place.  I don't think that code has been written reliably to get listings from a given sales order line.
         *  SalesOrderLine->getListings() needs some work.
         */
        //        if($listings->count() == 1)
        //        {
        //            /** @var ProductListing $listing */
        //            $listing = $listings->first();
        //
        //            if ($listing) {
        //                $mappings = collect(ProductMappingDto::from([
        //                    'listing_sku' => $listing->listing_sku,
        //                    'mapped_sku' => $salesOrderLine->sku,
        //                    'document_type' => $listing->document_type,
        //                ]));
        //
        //                $this->dispatch(new MapSalesOrderLinesJob($mappings, app($listing->document_type), $integrationInstance));
        //            }
        //        }

        return $this->response->setMessage(__('messages.sales_order.line_mapped'));
    }

    /**
     * Imports Sales Order Lines from CSV file
     *
     *
     * @throws Throwable
     */
    public function importLineItemsCsv(Request $request): Response
    {
        $request->validate([
            'file' => 'required_without:csvString',
            'csvString' => 'required_without:file',
            'sales_order_id' => 'nullable',
            'replace' => 'nullable|boolean', ]);

        $separator = $request->input('separator', ',');
        $escape = $request->input('escape', '"');
        $syncLines = $request->input('replace', false);
        // purchase order id or supplier id
        /** @var SalesOrder|null $salesOrder */
        $salesOrder = $request->input('sales_order_id') ? SalesOrder::query()->findOrFail($request->input('sales_order_id')) : null;

        // csv data form file or from string
        if ($request->input('file')) {
            $filePath = storage_path(config('uploader.models.target').$request->input('file'));
            $csvData = Helpers::csvFileToArray($filePath, $separator, $escape);
            $header = array_keys($csvData->current());
        } else {
            $csvData = Helpers::csvToArray($request->input('csvString'), $separator, $escape);
            $header = array_keys($csvData[0] ?? []);
        }

        // Headers require to be in CSV file
        $headers = ['sku'];
        if (count(array_intersect($header, $headers)) === 0) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError(__('messages.import_export.mismatch_template'), Response::CODE_INVALID_Template, 'file');
        }

        $skuArr = [];
        foreach ($csvData as $line) {
            // exclude rows that have sku and description are empty
            if (empty(trim($line['sku'])) && empty(trim($line['description']))) {
                continue;
            }

            $product = Product::query()->where('sku', $line['sku'])->first(['id', 'name', 'sku']);

            // price
            $price = null;
            if (empty($line['price'])) {
                if ($product) {
                    $productPricingQuery = ProductPricing::query()->where('product_id', $product->id);

                    if (empty($line['pricing_tier'])) {
                        // from the default pricing tier
                        $productPricingQuery->whereRelation('productPricingTier', 'is_default', true);
                    } else {
                        // from the selected pricing tier
                        $productPricingQuery->whereRelation('productPricingTier', 'name', $line['pricing_tier']);
                    }
                    $price = $productPricingQuery->value('price');
                }
            } else {
                $price = $line['price'];
            }

            $key = empty($line['sku']) ? $line['description'] : $line['sku'];
            $skuArr[$key]['quantity'] = ($skuArr[$key]['quantity'] ?? 0) + (empty($line['quantity']) ? 1 : $line['quantity']);
            $skuArr[$key]['amount'] = $price ?? ($skuArr[$key]['amount'] ?? null);
            $skuArr[$key]['sku'] = $line['sku'];
            $skuArr[$key]['product_id'] = $product?->id;
            $skuArr[$key]['is_product'] = (bool) $product;
            $skuArr[$key]['description'] = empty($line['description']) ? $product?->name : $line['description'];
            $skuArr[$key]['warehouse_id'] = $line['warehouse_id'] ?? $skuArr[$key]['warehouse_id'] ?? null;
            $skuArr[$key]['warehouse_name'] = $line['warehouse_name'] ?? $skuArr[$key]['warehouse_name'] ?? null;
        }

        if (empty($skuArr)) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.csv_no_records'));
        }

        // create and validate sales order update request
        $updateRequest = SalesOrderRequest::createFromCustom(['sales_order_lines' => array_values($skuArr)], 'PUT', ['sales_order' => $salesOrder ?: (new SalesOrder())]);
        $inputs = $updateRequest->validated();

        if ($salesOrder) {
            // add lines to the sales order
            DB::transaction(function () use ($syncLines, $inputs, $salesOrder) {
                $salesOrder->setSalesOrderLines($inputs['sales_order_lines'], true, $syncLines ? ['is_product' => true] : false);
            });
        } else {
            // lines to SalesOrderLines
            $salesOrderLines = [];
            // to load relations as bulk
            $products = Product::with(['parent', 'primaryImage', 'totalInventory', 'productInventory', 'inWarehouseAvailableQuantity'])->whereIn('id', array_column($skuArr, 'product_id'))->get();
            foreach ($skuArr as $line) {
                $salesOrderLine = new SalesOrderLine(Arr::except($line, ['sku', 'product']));
                $salesOrderLine->setRelation('product', $products->firstWhere('id', $line['product_id']));
                // set warehouse
                if ($salesOrderLine->product && empty($salesOrderLine->warehouse_id)) {
                    $warehouseRepository = app()->make(WarehouseRepository::class);
                    $salesOrderLine->warehouse_id = $warehouseRepository->getPriorityWarehouseIdForProduct($salesOrderLine->product, $salesOrderLine->quantity);
                    if (empty($salesOrderLine->warehouse_id)) {
                        $priorityWarehouses = json_decode(Setting::getValueByKey(Setting::KEY_WAREHOUSE_PRIORITY), true);
                        $salesOrderLine->warehouse_id = $priorityWarehouses[0] ?? null;
                    }
                }
                $salesOrderLine->load('warehouse');
                $salesOrderLines[] = $salesOrderLine;
            }
            // map to PurchaseOrderLineResource and set the included attributes
            $included = ['item_quantity', 'item_sku', 'item_barcode', 'item_price', 'item_name', 'is_product', 'is_bundle', 'is_variation', 'matrix_product', 'warehouse'];
            $request->merge(['included' => json_encode($included)]);
            $this->response->addData(SalesOrderLineResource::collection($salesOrderLines));
        }

        return $this->response->setMessage(__('messages.success.import', ['resource' => 'sales order lines']));
    }

    /**
     * Download importing lines template.
     */
    public function exportLines($id): BinaryFileResponse
    {
        $lines = collect();

        SalesOrderLine::with(['product', 'salesOrder'])
            ->where('sales_order_id', $id)
            ->each(function (SalesOrderLine $salesOrderLine) use ($lines) {
                $line = [
                    'sales_order_number' => $salesOrderLine->salesOrder->sales_order_number,
                    'sku' => $salesOrderLine->product?->sku,
                    'description' => $salesOrderLine->description ?: $salesOrderLine->product?->name,
                    'quantity' => $salesOrderLine->quantity,
                    'price' => $salesOrderLine->amount,
                    'warehouse_id' => $salesOrderLine->warehouse_id,
                ];
                $lines->add($line);
            });

        $headers = ['sales_order_number', 'sku', 'description', 'quantity', 'price', 'warehouse_id']; // static to return them when lines are empty

        $exportedFile = ExcelHelper::array2csvfile('sales_order_lines.csv', $headers, $lines);

        return response()->download($exportedFile)->deleteFileAfterSend(true);
    }

    /**
     * @throws Throwable
     */
    public function convertToFinancialLine(Request $request, SalesOrderLine $salesOrderLine, FinancialLineType $financialLineType): Response
    {
        app(SalesOrderManager::class)->convertToFinancialLine($salesOrderLine, $financialLineType);
        return $this->response->setMessage('Sales order line successfully converted to revenue line');
    }
}
