<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Exceptions\CantDeleteShippingProviderOrderAlreadyFulfilled;
use App\Exceptions\NotIntegratedException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Ghostscript\Ghostscript;
use App\Http\Controllers\Traits\BulkDeleteProcessor;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\FulfillSalesOrderRequest;
use App\Http\Resources\FulfillmentResource;
use App\Http\Resources\SalesOrderFulfillmentResource;
use App\Jobs\GenerateSalesOrderFulfillmentPackingSlipsJob;
use App\Models\PackingSlipQueue;
use App\Models\SalesOrderFulfillment;
use App\Observers\AddPackingSlipQueueObserver;
use App\Response;
use App\SDKs\ShipStation\ShipStationException;
use App\SDKs\Starshipit\StarshipitException;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use App\Services\ShippingProvider\RequestTrackingUpdatesService;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\Response as ResponseAlias;
use Throwable;

class SalesOrderFulfillmentController extends Controller
{
    use BulkDeleteProcessor, DataTable;
    use BulkOperation {
        deleteInstance as deleteInstanceOriginal;
    }

    protected $model_path = SalesOrderFulfillment::class;

    public function __construct(private readonly FulfillmentManager $manager)
    {
        parent::__construct();
    }

    /**+
     * View a sales order fulfillment
     *
     * @param SalesOrderFulfillment $salesOrderFulfillment
     *
     * @return Response
     */
    public function show(SalesOrderFulfillment $salesOrderFulfillment)
    {
        $salesOrderFulfillment->load('salesOrderFulfillmentLines.salesOrderLine.product');

        return $this->response->addData(SalesOrderFulfillmentResource::make($salesOrderFulfillment));
    }

    /**
     * Update a sales order fulfillment.
     *
     *
     * @throws Throwable
     */
    public function update(FulfillSalesOrderRequest $request, SalesOrderFulfillment $salesOrderFulfillment): Response
    {
        try {
            $this->manager->updateFulfillment($salesOrderFulfillment, $request->validated());

            return $this->response->success(ResponseAlias::HTTP_OK)
                ->setMessage(__('messages.success.update', ['resource' => 'sales order fulfillment']));
        } catch (SalesOrderFulfillmentException $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage())
                ->setErrors($e->errors);
        }
    }

    /**
     * Archive the sales order.
     */
    public function archive($salesOrderFulfillmentId): JsonResponse
    {
        $salesOrderFulfillment = SalesOrderFulfillment::with([])->findOrFail($salesOrderFulfillmentId);

        if ($salesOrderFulfillment->archive()) {
            return $this->response->setMessage(__('messages.success.archive', [
                'resource' => 'sales order fulfillment',
                'id' => $salesOrderFulfillment->fulfillment_number,
            ]));
        }

        return $this->response->addWarning(__('messages.failed.already_archive', ['resource' => 'sales order fulfillment', 'id' => $salesOrderFulfillment->fulfillment_number]), 'SalesOrderFulfillment'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $salesOrderFulfillmentId]);
    }

    public function unarchived($salesOrderFulfillmentId)
    {
        $salesOrderFulfillment = SalesOrderFulfillment::with([])->findOrFail($salesOrderFulfillmentId);

        if ($salesOrderFulfillment->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'sales order fulfillment',
                    'id' => $salesOrderFulfillment->fulfillment_number,
                ]));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'sales order fulfillment',
                'id' => $salesOrderFulfillment->fulfillment_number,
            ]), 'SalesOrderFulfillment'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $salesOrderFulfillment->id]);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:sales_order_fulfillments,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $fulfillments = SalesOrderFulfillment::with(['salesOrder'])->whereIn('id', $ids)->get();
        foreach ($fulfillments as $index => $fulfillment) {
            $isUsed = $fulfillment->isUsed();

            $result[$index] = $fulfillment->only('id', 'fulfillment_number');
            $result[$index]['deletable'] = $isUsed ? false : true;
            $result[$index]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * @throws Throwable
     */
    public function destroy(Request $request, $fulfillmentId): Response
    {
        $checkDeletable = $request->input('check-deletable', 1);
        /** @var SalesOrderFulfillment $fulfillment */
        $fulfillment = SalesOrderFulfillment::query()->findOrFail(e($fulfillmentId));

        if ($checkDeletable && $used = $fulfillment->isUsed()) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError($used['status'], 'DeleteShippingProviderOrder'.Response::CODE_UNACCEPTABLE, 'id');
        }

        try {
            $fulfillment->delete(! $checkDeletable);
        } catch (CantDeleteShippingProviderOrderAlreadyFulfilled|UnableToCancelOutboundFulfillmentOrderException $exception) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError($exception->getMessage(), 'DeleteShippingProviderOrder'.Response::CODE_UNACCEPTABLE, 'id');
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'sales order fulfillment',
            'id' => $fulfillment->fulfillment_number,
        ]));
    }

    /**
     * @param Request $request
     * @param string $fulfillmentType
     *
     * @return Response
     */
    public function report(Request $request, string $fulfillmentType)
    {
        $from = $request->query('from');
        $to = $request->query('to');

        $query = SalesOrderFulfillment::query()
            ->selectRaw('Date(`created_at`) as `fulfilled_date`, count(`id`) as `fulfilled_quantity`')
            ->where('fulfillment_type', $fulfillmentType)
            ->groupBy('fulfilled_date');

        if ($from) {
            $query->whereDate('created_at', '>=', $from);
        }
        if ($to) {
            $query->whereDate('created_at', '<=', $to);
        }

        return $this->response->addData($query->get());
    }

    /**
     * Export packing slips of sales order fulfillments.
     *
     *
     * @throws Exception
     */
    public function exportPackingSlips(Request $request): Response
    {
        $request->validate([
            'ids' => 'required',
        ]);

        $ids = explode(',', $request->get('ids'));
        $printedFulfillmentIds = $ids;

        // Make sure the fulfillment ids exist
        $ids = array_filter($ids, function ($id) {
            return SalesOrderFulfillment::query()->where('id', $id)->exists();
        });

        $needsPrinting = [];
        foreach ($ids as $id) {
            // the file does not exist
            if (! Storage::disk('fulfillment_packing_slips')->exists("$id.pdf")) {
                $needsPrinting[] = $id;
                AddPackingSlipQueueObserver::addPackingSlipQueue((new SalesOrderFulfillment())->forceFill(['id' => (int) $id]), false);
            }
            // generating the packing slip still in-progress (in the queue)
            elseif (PackingSlipQueue::query()->firstWhere(['link_id' => (int) $id, 'link_type' => SalesOrderFulfillment::class])) {
                $needsPrinting[] = $id;
            }
        }

        // print unprinted orders in one file
        if (! empty($needsPrinting)) {
            $unprintedFile = (new GenerateSalesOrderFulfillmentPackingSlipsJob($needsPrinting, null, ['one_file' => true]))->handle();
            $unprintedFile = str_replace([Storage::disk('fulfillment_packing_slips')->path(''), '.pdf'], '', $unprintedFile);
            $ids = array_merge(array_diff($ids, $needsPrinting), [$unprintedFile]);
        }
        $ids = array_unique($ids);

        if (empty($ids)) {
            return $this->response->setMessage('There are no packing slips for the selected fulfillments.');
        }

        // mark sales order fulfillments as printed
        SalesOrderFulfillment::query()->whereIn('id', $printedFulfillmentIds)->update(['packing_slip_printed_at' => now()]);

        if (count($ids) === 1) {
            return $this->response->addData(['file' => Storage::disk('fulfillment_packing_slips')->url("{$ids[0]}.pdf")]);
        }
        $outputFile = time().rand(1000, 9999).'.pdf';
        $psFiles = array_map(fn ($id) => Storage::disk('fulfillment_packing_slips')->path("$id.pdf"), $ids);

        if (is_null($combineStatus = ( new Ghostscript(Storage::disk('reports')->path($outputFile), $psFiles) )->combine())) {
            return $this->response->addData(['file' => Storage::disk('reports')->url($outputFile)]);
        }

        throw new Exception("Ghostscript error: {$combineStatus}");
    }

    /**
     * bulk delete using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        $request->validate([
            'filters' => 'required_without:ids',
            'ids' => 'required_without:filters|array|min:1',
            'ids.*' => 'integer|exists:sales_order_fulfillments,id',
        ]);

        if (! $this->isLargeDelete(SalesOrderFulfillment::class, $request)) {
            return $this->bulkOperation($request, $this->BULK_DELETE);
        }

        // Attempting to deleting a large set of records,
        // we will do it in the background.
        $this->processLargeBulkDelete(SalesOrderFulfillment::class, $request);

        return $this->response->setMessage('Added to the Queue, it will be process shortly');
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Bulk request tracking updates from the shipping providers
     */
    public function bulkRequestTrackingUpdates(Request $request): Response
    {
        try {
            $requestService = (new RequestTrackingUpdatesService($request))->execute();
        } catch (StarshipitException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError("Starshipit response: {$e->getMessage()}", Response::CODE_INTEGRATION_API_ERROR, 'request', ['code' => $e->getCode(), 'message' => $e->getMessage()]);
        } catch (ShipStationException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError("Shipstation response: {$e->getMessage()}", Response::CODE_INTEGRATION_API_ERROR, 'request', ['code' => $e->getCode(), 'message' => $e->getMessage()]);
        } catch (NotIntegratedException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError($e->getMessage(), Response::CODE_NOT_INTEGRATED, 'request');
        }

        return $this->response->setMessage(__($requestService->queued ? 'messages.success.queued' : 'messages.sales_order.tracking_updated'));
    }

    /**
     * Mark the sales order fulfillment as printed
     */
    public function markAsPrinted($salesOrderFulfillmentId): Response
    {
        SalesOrderFulfillment::where('id', $salesOrderFulfillmentId)->update(['packing_slip_printed_at' => now()]);

        return $this->response->setMessage(__('messages.sales_order.marked_as_printed'));
    }

    protected function deleteInstance(Model $instance)
    {
        try {
            return $this->deleteInstanceOriginal($instance);
        } catch (CantDeleteShippingProviderOrderAlreadyFulfilled|UnableToCancelOutboundFulfillmentOrderException $exception) {
            $this->response->addWarning($exception->getMessage(), 'DeleteShippingProviderOrder'.Response::CODE_UNACCEPTABLE, "ids.{$instance->id}.id", [
                'resource' => $this->getModelName($this->model_path),
                'resource_id' => $instance->id,
            ]);

            return false;
        }
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return SalesOrderFulfillment::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return FulfillmentResource::class;
    }
}
