<?php

namespace App\Http\Controllers;

use App\Http\Resources\PaymentMethodMappingSalesChannelResource;
use App\Http\Resources\ShippingMethodMappingSalesChannelResource;
use App\Models\IntegrationInstance;
use App\Models\PaymentMethodMappingSalesChannelToSku;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\ShippingMethodMappingsSalesChannelToSku;
use App\Models\ShippingMethodMappingsSkuToShippingProviderMethod;
use App\Response;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class SalesChannelController extends Controller
{
    /**
     * Retrieve list of sales channel mapping.
     */
    public function shippingMethods($salesChannelId): AnonymousResourceCollection
    {
        $salesChannelMappings = ShippingMethodMappingsSalesChannelToSku::with(['shippingMethod'])->where('sales_channel_id', $salesChannelId)->get();

        return ShippingMethodMappingSalesChannelResource::collectionWithTableSpecifications($salesChannelMappings, null);
    }

    /**
     * Get sales orders count that used shipping method.
     */
    public function shippingMethodSalesOrdersCount(Request $request, $salesChannelId): Response
    {
        $request->validate(['shipping_method' => 'required']);

        $salesOrdersCount = SalesOrder::with([])
            ->where('sales_channel_id', $salesChannelId)
            ->where('requested_shipping_method', $request->input('shipping_method'))
            ->count();

        return $this->response->setMessage(__('messages.sales_order.mapping_shipping_methods_count', [
            'salesOrdersCount' => $salesOrdersCount,
            'service_name' => $request->input('shipping_method'),
        ]));
    }

    /**
     * Map sales channel's shipping methods.
     */
    public function mapShippingMethods(Request $request, $salesChannelId): Response
    {
        $request->validate([
            'mappings' => 'required|array|min:1',
            'mappings.*.id' => 'required_without:mappings.*.sales_channel_method|exists:shipping_method_mappings_sales_channel_to_sku,id',
            'mappings.*.sales_channel_method' => 'required_without:mappings.*.id',
            'mappings.*.shipping_method_id' => 'nullable|exists:shipping_methods,id',
        ]);

        foreach ($request->input('mappings') as $mapping) {
            // save mapping
            if (isset($mapping['id'])) {
                $salesChannelMapping = ShippingMethodMappingsSalesChannelToSku::with([])->findOrFail($mapping['id']);
            } else {
                $salesChannelMapping = ShippingMethodMappingsSalesChannelToSku::with([])
                    ->firstOrCreate(['sales_channel_method' => $mapping['sales_channel_method'], 'sales_channel_id' => $salesChannelId]);
            }
            $salesChannelMapping->shipping_method_id = $mapping['shipping_method_id'] ?? null;
            $salesChannelMapping->save();

            // reflect it on sales orders
            SalesOrder::with([])->where('sales_channel_id', $salesChannelId)
                ->where('order_status', '!=',SalesOrder::STATUS_CLOSED)
                ->where('requested_shipping_method', $salesChannelMapping->sales_channel_method)
                ->update(['shipping_method_id' => $salesChannelMapping->shipping_method_id]);
        }

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'shipping method mappings']));
    }

    public function unmapShippingMethod($salesChannelId, string $salesChannelMethodName): Response
    {

        $mapping = ShippingMethodMappingsSalesChannelToSku::with([])
            ->where('sales_channel_id', $salesChannelId)
            ->where('sales_channel_method', $salesChannelMethodName)
            ->first();

        if (!$mapping) {
            return $this->response->addError(__('messages.error.not_found', ['resource' => 'shipping method mapping']), Response::CODE_NOT_FOUND, 'shipping_provider_method');
        }

        // Remove fulfillment mappings
        $shippingMethod = $mapping->shippingMethod;
        SalesOrder::with([])->where('sales_channel_id', $salesChannelId)
            ->where('order_status', '!=',SalesOrder::STATUS_CLOSED)
            ->where('requested_shipping_method', $mapping->sales_channel_method)
            ->update(['shipping_method_id' => null]);

        $mapping->delete();

        return $this->response->setMessage(__('messages.success.delete', ['resource' => 'shipping method mapping', 'id' => $shippingMethod?->id]));
    }

    /**
     * Retrieve shipping method mappings of the sales channel.
     */
    public function paymentMethods($salesChannelId): AnonymousResourceCollection
    {
        $salesChannelMappings = PaymentMethodMappingSalesChannelToSku::with(['paymentType'])->where('sales_channel_id', $salesChannelId)->get();

        return PaymentMethodMappingSalesChannelResource::collectionWithTableSpecifications($salesChannelMappings, null);
    }

    /**
     * Map sales channel's payment methods.
     *
     * @param $salesChannelId
     */
    public function mapPaymentMethods(Request $request, SalesChannel $salesChannel): Response
    {
        $request->validate([
            'mappings' => 'required|array|min:1',
            'mappings.*.id' => 'required_without:mappings.*.sales_channel_method|exists:payment_method_mappings_sales_channel_to_sku,id',
            'mappings.*.sales_channel_method' => 'required_without:mappings.*.id',
            'mappings.*.payment_type_id' => 'nullable|exists:payment_types,id',
        ]);

        foreach ($request->input('mappings') as $mapping) {
            // save mapping
            if (isset($mapping['id'])) {
                $paymentMapping = PaymentMethodMappingSalesChannelToSku::query()->findOrFail($mapping['id']);
            } else {
                $paymentMapping = PaymentMethodMappingSalesChannelToSku::query()->firstOrCreate(['sales_channel_id' => $salesChannel->id, 'sales_channel_method' => $mapping['sales_channel_method']]);
            }
            $paymentMapping->payment_type_id = $mapping['payment_type_id'] ?? null;
            $paymentMapping->save();

            // TODO: map sales orders' payments
        }

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'payment method mappings']));
    }

    /**
     * Delete the payment method mapping
     */
    public function destroyPaymentMethodMapping($paymentMethodMappingId): Response
    {
        $mapping = PaymentMethodMappingSalesChannelToSku::query()->findOrFail($paymentMethodMappingId);
        $mapping->delete();

        return $this->response->setMessage(__('messages.success.delete', ['resource' => 'payment method mapping', 'id' => $mapping->sales_channel_method]));
    }

    /**
     * Delete the shipping mapping
     */
    public function destroyShippingMapping($shippingId): Response
    {
        $mapping = ShippingMethodMappingsSalesChannelToSku::query()->findOrFail($shippingId);
        $mapping->delete();

        return $this->response->setMessage(__('messages.success.delete', ['resource' => 'shipping method mapping', 'id' => $mapping->sales_channel_method]));
    }

    /**
     * Lookup on payment methods of the sales channel
     */
    public function paymentMethodsLookup(Request $request, $salesChannelId): Response
    {
        $value = $request->get('value');

        return $this->response->addData(PaymentMethodMappingSalesChannelToSku::query()
            ->where('sales_channel_id', $salesChannelId)
            ->whereNull('payment_type_id')
            ->where('sales_channel_method', 'like', "$value%")
            ->pluck('sales_channel_method'));
    }
}
