<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\StoreReturnReason;
use App\Http\Resources\ReturnReasonResource;
use App\Models\ReturnReason;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ReturnReasonController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = ReturnReason::class;

    private $resource = 'return reason';

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreReturnReason $request): Response
    {
        $returnReason = new ReturnReason($request->validated());
        $returnReason->save();

        return $this->response->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(ReturnReasonResource::make($returnReason));
    }

    /**
     * Display the specified resource.
     */
    public function show(ReturnReason $returnReason): Response
    {
        return $this->response->addData(ReturnReasonResource::make($returnReason));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(StoreReturnReason $request, ReturnReason $returnReason): Response
    {
        $returnReason->update($request->validated());

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $returnReason->name,
            ]))
            ->addData(ReturnReasonResource::make($returnReason));
    }

    /**
     * Remove the specified resource from storage.
     *
     *
     * @throws Exception
     */
    public function destroy(ReturnReason $returnReason): Response
    {
        $reasons = $returnReason->delete();

        // check if the payment term is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['return_reason_id' => $returnReason->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => $this->resource,
                    'id' => $returnReason->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $returnReason->name,
        ]));
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:'.( new $this->model_path )->getTable().',id',
        ]);

        $ids = array_unique($request->input('ids', []));
        $result = [];

        $instances = ReturnReason::with([])->withCount('salesCreditReturnLines')->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($instances as $key => $instance) {
            $isUsed = $instance->isUsed();

            $result[$key] = $instance->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * Archive the return reason.
     */
    public function archive(ReturnReason $returnReason): Response
    {
        if ($returnReason->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $returnReason->name,
                ]))
                ->addData(ReturnReasonResource::make($returnReason));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $returnReason->name,
            ]), 'ReturnReason'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $returnReason->id])
            ->addData(ReturnReasonResource::make($returnReason));
    }

    /**
     * Unarchived the return reason.
     */
    public function unarchived(ReturnReason $returnReason): Response
    {
        if ($returnReason->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $returnReason->name,
                ]))
                ->addData(ReturnReasonResource::make($returnReason));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $returnReason->name,
            ]), 'ReturnReason'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $returnReason->id])
            ->addData(ReturnReasonResource::make($returnReason));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Bulk delete return reasons.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return ReturnReason::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return ReturnReasonResource::class;
    }
}
