<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\DataTable\DataTableResource;
use App\Events\PurchaseOrderLineDeleted;
use App\Helpers;
use App\Helpers\ExcelHelper;
use App\Http\Requests\StorePurchaseOrder;
use App\Http\Resources\PurchaseOrderLineResource;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SupplierProductPricing;
use App\Response;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Throwable;

use function array_intersect;
use function array_keys;
use function collect;
use function config;
use function count;
use function response;
use function storage_path;

class PurchaseOrderLineController extends Controller
{
    use DataTable;

    public function getLines(
        Request $request,
        PurchaseOrder $purchaseOrder
    ): JsonResource|AnonymousResourceCollection {
        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return new JsonResource(DataTableConfiguration::getTableSpecifications((new PurchaseOrderLine())->dataTableKey));
        }

        /** @var Builder $builder */
        $builder = $this->buildIndexQuery($request)->where('purchase_order_id', $purchaseOrder->id);

        // TODO: I'm not sure the with here even works.  It is more for show methods rather than index based methods.
        $builder->with([
            'product',
            'product.totalInventory',
            'product.suppliersInventory',
            'product.productInventory',
            'product.images',
            'coveredBackorderQueues.backorderQueue.salesOrderLine.salesOrder',
            'coveredBackorderQueues.purchaseOrderLine.purchaseOrder',
            'purchaseOrderShipmentReceiptLines.backorderQueueReleases.backorderQueue.salesOrderLine.salesOrder',
            'purchaseOrderShipmentReceiptLines.backorderQueueReleases.link',
            'purchaseOrderShipmentReceiptLines.purchaseOrderLine.purchaseOrder',
        ]);

        $builder = DataTableConfiguration::paginate($builder);

        return $this->getResource()::collectionWithTableSpecifications($builder, $this->getModel());
    }

    /**
     * Delete purchase order line.
     */
    public function destroy($line_id): JsonResponse
    {
        $purchaseOrderLine = PurchaseOrderLine::withCount('purchaseInvoiceLine')->findOrFail($line_id);

        /**
         * Check if Invoice issued for the purchase order.
         */
        if ($purchaseOrderLine->purchase_invoice_line_count) {
            return response()->json([
                'status' => 'failed',
                'message' => 'Invoice issued for this purchase order line. You can\'t delete it.',
            ], 422);
        }

        /**
         * Check if last line of purchase order lines.
         */
        if (PurchaseOrderLine::where('purchase_order_id', $purchaseOrderLine->purchase_order_id)->count() == 1) {
            return response()->json([
                'status' => 'failed',
                'message' => 'Can\'t delete last line of a purchase order.',
            ], 422);
        }

        /**
         * Delete line.
         */
        $purchaseOrderLine->delete();

        /**
         * fire event
         */
        event(new PurchaseOrderLineDeleted($purchaseOrderLine));

        return response()->json(['status' => 'success']);
    }

    /**
     * Imports PO line items from CSV file
     *
     *
     * @throws Throwable
     */
    public function importLineItemsCsv(Request $request): Response
    {
        $request->validate([
            'file' => 'required_without:csvString',
            'csvString' => 'required_without:file',
            'purchase_order_id' => 'required_without:supplier_id',
            'supplier_id' => 'required_without:purchase_order_id',
            'replace' => 'nullable|boolean', ]);

        $separator = $request->input('separator', ',');
        $escape = $request->input('escape', '"');
        $syncLines = $request->input('replace', false);
        // purchase order id or supplier id
        $purchaseOrder = $request->input('purchase_order_id') ? PurchaseOrder::query()->findOrFail($request->input('purchase_order_id')) : null;
        $supplierId = $purchaseOrder ? $purchaseOrder->supplier_id : $request->input('supplier_id');

        // csv data form file or from string
        if ($request->input('file')) {
            $filePath = storage_path(config('uploader.models.target').$request->input('file'));
            $csvData = Helpers::csvFileToArray($filePath, $separator, $escape);
            $header = array_keys($csvData->current());
        } else {
            $csvData = Helpers::csvToArray($request->input('csvString'), $separator, $escape);
            $header = array_keys($csvData[0] ?? []);
        }

        // Headers require to be in CSV file
        $headers = ['sku'];
        if (count(array_intersect($header, $headers)) === 0) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError(__('messages.import_export.mismatch_template'), Response::CODE_INVALID_Template, 'file');
        }

        $skuArr = [];
        foreach ($csvData as $line) {
            // exclude the blank sku
            if (empty(trim(@$line['sku'])) && empty(trim(@$line['barcode']))) {
                continue;
            }

            $product = Product::with(['parent'])->where('sku', $line['sku'])
                ->when(! empty($line['barcode']), function ($builder) use ($line) {
                    return $builder->orWhere('barcode', $line['barcode']);
                })
                ->first();

            if (! $product) {
                return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage('Product not found for '.$line['sku']);
            }

            $supplierProduct = $product->supplierProducts->where('supplier_id', $supplierId)->first();
            if ($product && $product->type != 'standard') {
                return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage('Only standard products can be purchased.');
            }
            if (! $product) {
                return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage('Product not found for '.$line['sku']);
            }
            $line['sku'] = $product->sku;
            // unit cost
            $unitCost = null;
            if (empty($line['unit_cost'])) {
                if ($product) {
                    $supplierProductPricingQuery = SupplierProductPricing::query();
                    $supplierProductPricingQuery->whereHas('supplierProduct', function (Builder $builder) use ($supplierId, $product) {
                        $builder->where('product_id', $product->id);
                        $builder->where('supplier_id', $supplierId);
                    });

                    if (empty($line['pricing_tier'])) {
                        // from default pricing tier
                        $supplierProductPricingQuery->whereRelation('supplierPricingTier', 'is_default', true);
                    } else {
                        // from pricing tier
                        $supplierProductPricingQuery->whereRelation('supplierPricingTier', 'name', $line['pricing_tier']);
                    }
                    $unitCost = $supplierProductPricingQuery->value('price');
                }
            } else {
                $unitCost = $line['unit_cost'];
            }

            $skuArr[$line['sku']]['quantity'] = ($skuArr[$line['sku']]['quantity'] ?? 0) + (empty($line['quantity']) ? 1 : $line['quantity']);
            $skuArr[$line['sku']]['amount'] = $unitCost ?: ($skuArr[$line['sku']]['amount'] ?? 0);
            $skuArr[$line['sku']]['sku'] = ! empty($line['sku']) ? $line['sku'] : $product->sku;
            $skuArr[$line['sku']]['product'] = $product;
            $skuArr[$line['sku']]['description'] = $line['description'] ?? $product->name ?? null;
            if (isset($line['discount_rate']) && $line['discount_rate'] != '') {
                $skuArr[$line['sku']]['discount'] = $line['discount_rate'] * 100;
            }
            if (isset($line['tax_rate_id']) && $line['tax_rate_id'] != '') {
                $skuArr[$line['sku']]['tax_rate_id'] = $line['tax_rate_id'];
            } else {
                $tax_rate_id = $supplierProduct->default_tax_rate_id
                    ?? $product->default_tax_rate_id
                    ?? $purchaseOrder->supplier->default_tax_rate_id
                    ?? null;

                $skuArr[$line['sku']]['tax_rate_id'] = $tax_rate_id;
            }
        }

        if (empty($skuArr)) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.csv_no_records'));
        }

        // create and validate purchase order update request
        $updateRequest = StorePurchaseOrder::createFromCustom(['purchase_order_lines' => array_values($skuArr)], 'PUT', ['purchase_order' => $purchaseOrder ?: (new PurchaseOrder(['supplier_id' => $supplierId]))]);
        $inputs = $updateRequest->validated();
        if ($purchaseOrder) {
            // add lines to the purchase order
            DB::transaction(function () use ($syncLines, $inputs, $purchaseOrder) {
                $purchaseOrder->setPurchaseOrderLines($inputs['purchase_order_lines'], $syncLines);
            });
        } else {
            // lines to PurchaseOrderLines
            $purchaseOrderLines = [];
            foreach ($skuArr as $line) {
                $purchaseOrderLine = new PurchaseOrderLine(Arr::except($line, ['sku', 'product']));
                $purchaseOrderLine->nominal_code_id = $purchaseOrderLine->getNominalCodeId($line['product']);
                $purchaseOrderLine->product_id = $line['product']?->id;
                $purchaseOrderLine->setRelation('product', $line['product']);
                $purchaseOrderLines[] = $purchaseOrderLine;
            }
            // map to PurchaseOrderLineResource and set the included attributes
            $included = ['quantity', 'sku', 'barcode', 'is_variation', 'matrix_product', 'price', 'name', 'nominal_code_id'];
            $request->merge(['included' => json_encode($included)]);
            $this->response->addData(PurchaseOrderLineResource::collection($purchaseOrderLines));
        }

        return $this->response->setMessage(__('messages.success.import', ['resource' => 'purchase order lines']));
    }

    /**
     * Download importing lines template.
     */
    public function exportLines($id): BinaryFileResponse
    {
        $lines = collect();

        PurchaseOrderLine::with(['product', 'purchaseOrder'])
            ->where('purchase_order_id', $id)
            ->each(function (PurchaseOrderLine $purchaseOrderLine) use ($lines) {
                $line = [
                    'purchase_order_number' => $purchaseOrderLine->purchaseOrder->purchase_order_number,
                    'sku' => $purchaseOrderLine->product?->sku,
                    'barcode' => $purchaseOrderLine->product?->barcode,
                    'description' => $purchaseOrderLine->description ?: $purchaseOrderLine->product?->name,
                    'quantity' => $purchaseOrderLine->quantity,
                    'unit_cost' => $purchaseOrderLine->amount,
                    'discount_rate' => $purchaseOrderLine->discount_rate,
                    'tax_rate_id' => $purchaseOrderLine->tax_rate_id,
                ];
                $lines->add($line);
            });
        $headers = [
            'purchase_order_number',
            'sku',
            'barcode',
            'description',
            'quantity',
            'unit_cost',
            'discount_rate',
            'tax_rate_id',
        ]; // static to return them when lines are empty

        $exportedFile = ExcelHelper::array2csvfile('purchase_order_lines.csv', $headers, $lines);

        return response()->download($exportedFile)->deleteFileAfterSend(true);
    }

    protected function getModel()
    {
        return PurchaseOrderLine::class;
    }

    protected function getResource(): DataTableResource|string
    {
        return PurchaseOrderLineResource::class;
    }
}
