<?php

namespace App\Http\Controllers;

use App\Exporters\PurchaseOrderBuilderExporter;
use App\Http\Requests\PurchaseOrderBuilderRequest;
use App\Http\Resources\POBuilderSalesOrderLineInventoryResource;
use App\Http\Resources\PurchaseOrderBuilderLineResource;
use App\Models\PurchaseOrder;
use App\Response;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Exceptions\LargePurchaseOrderException;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilder;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilderFactory;
use Illuminate\Contracts\Container\BindingResolutionException;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

/**
 * Class PurchaseOrderBuilderController.
 */
class PurchaseOrderBuilderController extends Controller
{
    /**
     * @throws BindingResolutionException
     */
    public function build(PurchaseOrderBuilderRequest $request): Response
    {
        /** @see SKU-4423 */
        set_time_limit(60 * 6);

        $payload = $request->validated();

        try {
            POBuilderSalesOrderLineInventoryResource::$warehouseId = $payload['destination_warehouse_id'] ?? null;

            return $this->response->addData(
                PurchaseOrderBuilderLineResource::collection(PurchaseOrderBuilderFactory::make(
                    $payload['forecast_type'],
                    array_merge($payload['options'], [
                        'supplier_id' => $payload['supplier_id'],
                        'destination_warehouse_id' => POBuilderSalesOrderLineInventoryResource::$warehouseId,
                    ])
                )->build())
            );
        } catch (LargePurchaseOrderException $e) {
            return $this->response->error()->addError(
                "Purchase order involves {$e->count} product(s) which exceeds the allowable limit of " . PurchaseOrderBuilder::MAX_PRODUCTS_COUNT . '. Please filter down results.',
                Response::CODE_UNACCEPTABLE,
                'PurchaseOrderBuilder'
            );
        }
    }

    /**
     * @throws BindingResolutionException
     */
    public function export(PurchaseOrderBuilderRequest $request, PurchaseOrder $purchaseOrder): Response
    {
        set_time_limit(60 * 12);

        $payload = $request->validated();

        try {
            
            $destinationWarehouseId = $payload['destination_warehouse_id'] ?? null;
            POBuilderSalesOrderLineInventoryResource::$warehouseId = $destinationWarehouseId;

            $data = PurchaseOrderBuilderFactory::make(
                $payload['forecast_type'],
                array_merge($payload['options'], [
                    'supplier_id' => $payload['supplier_id'],
                    'destination_warehouse_id' => $destinationWarehouseId,
                ])
            )->build();

            $export = new PurchaseOrderBuilderExporter($data, 'csv');
            $export->export();

            return $this->response->addData(route('export.download') . '?file=' . $export->getFilename());
        } catch (LargePurchaseOrderException $e) {
            return $this->response->error()->addError(
                "Purchase order involves {$e->count} product(s) which exceeds the allowable limit of " . PurchaseOrderBuilder::MAX_PRODUCTS_COUNT . '. Please filter down results.',
                Response::CODE_UNACCEPTABLE,
                'PurchaseOrderBuilder'
            );
        }
    }
}
