<?php

namespace App\Http\Controllers;

use App\Http\Requests\PurchaseInvoicePaymentRequest;
use App\Http\Resources\PaymentResource;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\PurchaseInvoice;
use App\Response;
use Carbon\Carbon;
use Illuminate\Http\Request;

class PurchaseInvoicePaymentController extends Controller
{
    public function index($purchaseInvoiceId): Response
    {
        $purchaseInvoice = PurchaseInvoice::with(['payments'])->findOrFail(e($purchaseInvoiceId));

        return $this->response->addData(PaymentResource::collection($purchaseInvoice->payments))->success(
            Response::HTTP_OK
        );
    }

    public function store(PurchaseInvoicePaymentRequest $request, $purchaseInvoiceId): Response
    {
        $purchaseInvoice = $this->findOrFailPurchaseInvoice($purchaseInvoiceId);
        $data = $request->validated();

        if (empty($data['payment_date'])) {
            $data['payment_date'] = Carbon::now()->format('Y-m-d');
        }

        // Add the payment to the purchase invoice
        /** @var Payment $payment */
        $payment = $purchaseInvoice->payments()->create(array_merge($data, [
            'currency_id' => $data['currency_id'] ?? $purchaseInvoice->currency_id,
        ]));

        return $this->paymentSaved($purchaseInvoice, $payment, $data['payment_date'], 'create');
    }

    public function update(PurchaseInvoicePaymentRequest $request, $purchaseInvoiceId, $paymentId): Response
    {
        $purchaseInvoice = $this->findOrFailPurchaseInvoice($purchaseInvoiceId);

        /** @var Payment $payment */
        $payment = $purchaseInvoice->payments()->findOrFail(e($paymentId));

        $data = $request->validated();

        // Credit payment types cannot be changed as they are system created
        if ($payment->paymentType->name === PaymentType::SALES_CREDIT_PAYMENT_TYPE_NAME) {
            unset($data['payment_type_id']);
        }
        $payment->fill($data);
        $payment->save();

        return $this->paymentSaved($purchaseInvoice, $payment, $data['payment_date'] ?? null, 'update');
    }

    /**
     * @param  null  $paymentDate
     */
    private function paymentSaved(PurchaseInvoice $purchaseInvoice, Payment $payment, $paymentDate, string $action): Response
    {
        // We update the payment status of the purchase invoice
        $purchaseInvoice->setPaymentStatus();

        if ($action !== strtolower(Request::METHOD_DELETE)) {
            $this->response->addData($payment);
        }

        return $this->response->setMessage(__("messages.success.{$action}", [
            'resource' => 'purchase invoice payment',
            'id' => $payment->id,
        ]));
    }

    public function destroy($purchaseInvoiceId, $paymentId)
    {
        $purchaseInvoice = $this->findOrFailPurchaseInvoice(e($purchaseInvoiceId));

        /** @var Payment $payment */
        $payment = $purchaseInvoice->payments()->findOrFail(e($paymentId));

        // Delete the payment
        $payment->delete();

        return $this->paymentSaved($purchaseInvoice, $payment, null, 'delete');
    }

    private function findOrFailPurchaseInvoice($id, array $relations = ['purchaseInvoiceLines']): PurchaseInvoice
    {
        return PurchaseInvoice::with($relations)->findOrFail(e($id));
    }
}
