<?php

namespace App\Http\Controllers;

use App\Http\Resources\ProductAttributeResource;
use App\Http\Resources\ProductResource;
use App\Models\Product;
use App\Response;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;

/**
 * Class ProductVariationController.
 */
class ProductVariationController extends Controller
{
    public function variationsForProduct(Product $product): Response
    {
        if ($product->type !== Product::TYPE_MATRIX) {
            return $this->respondNotMatrixProduct($product);
        }

        return $this->response->addData(
            ProductResource::collection($product->variations)
        );
    }

    public function attributes(Product $product): Response
    {
        if ($product->type !== Product::TYPE_MATRIX) {
            return $this->respondNotMatrixProduct($product);
        }

        $variantAttributes = $this->getUniqueAttributes($product->variations);

        return $this->response->addData(
            ProductAttributeResource::collection($variantAttributes)
        );
    }

    public function allForProducts(Request $request): Response
    {
        if (! $request->has('product_ids')) {
            return $this->response->addError(
                'Product ids must be provided',
                Response::HTTP_BAD_REQUEST,
                'ProductIdsRequired'
            )->error(Response::HTTP_BAD_REQUEST);
        }

        $attributes = $this->getUniqueAttributes(
            Product::with(['productAttributeValues'])
                ->whereIn('id', json_decode($request->query('product_ids')))
                ->get()
        );

        return $this->response->addData(
            ProductAttributeResource::collection($attributes)
        );
    }

    /**
     * @param  Collection|mixed  $products
     * @return Collection|\Illuminate\Support\Collection|mixed
     */
    private function getUniqueAttributes(Collection $products)
    {
        return $products->map(function (Product $product) {
            return $product->productAttributeValues;
        })->flatten()
            ->unique(function ($value) {
                return $value->attribute_id;
            });
    }

    private function respondNotMatrixProduct(Product $product): Response
    {
        return $this->response->addError(
            __('messages.product.not_matrix', [
                'id' => $product->id,
            ]),
            Response::HTTP_BAD_REQUEST,
            'ProductIdNotMatrix'
        )->error(Response::HTTP_BAD_REQUEST);
    }
}
